package ru.yandex.kikimr.util;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class NameRange {
    @Nullable
    private final String begin;
    @Nullable
    private final String end;
    private final boolean beginInclusive;
    private final boolean endInclusive;

    public NameRange(@Nullable String begin, boolean beginInclusive, @Nullable String end, boolean endInclusive) {
        this.begin = begin;
        this.end = end;
        this.beginInclusive = beginInclusive;
        this.endInclusive = endInclusive;

        if (begin != null && end != null) {
            int cmp = end.compareTo(begin);
            if (cmp > 0) {
                // OK
            } else if (cmp < 0) {
                throw new IllegalArgumentException("end < begin");
            } else if (cmp == 0) {
                if (beginInclusive && endInclusive) {
                    // OK
                } else {
                    throw new IllegalArgumentException("end == begin && !inclusive both");
                }
            } else {
                throw new IllegalStateException();
            }
        }
    }

    public NameRange from(String begin, boolean beginInclusive) {
        if (this.begin != null) {
            throw new RuntimeException("not comparable");
        }
        return new NameRange(begin, beginInclusive, end, endInclusive);
    }

    public NameRange gt(String begin) {
        return from(begin, false);
    }

    public NameRange ge(String begin) {
        return from(begin, true);
    }

    public NameRange to(String end, boolean endInclusive) {
        if (this.end != null) {
            throw new RuntimeException("not comparable");
        }
        return new NameRange(begin, beginInclusive, end, endInclusive);
    }

    public NameRange lt(String end) {
        return to(end, false);
    }

    public NameRange le(String end) {
        return to(end, true);
    }

    public static NameRange all() {
        return new NameRange(null, false, null, false);
    }

    public static NameRange inclusive(String begin, String end) {
        return new NameRange(begin, true, end, true);
    }

    static NameRange exclusive(String begin, String end) {
        return new NameRange(begin, false, end, false);
    }

    public static NameRange inclusiveExclusive(String begin, String end) {
        return new NameRange(begin, true, end, false);
    }

    public static NameRange rangeTo(String end, boolean inclusive) {
        return new NameRange(null, false, end, inclusive);
    }

    public static NameRange rangeFrom(String begin, boolean inclusive) {
        return new NameRange(begin, inclusive, null, false);
    }

    public static NameRange single(String single) {
        return inclusive(single, single);
    }

    public boolean includes(String name) {
        if (begin != null) {
            if (beginInclusive) {
                if (begin.compareTo(name) > 0) {
                    return false;
                }
            } else {
                if (begin.compareTo(name) >= 0) {
                    return false;
                }
            }
        }

        if (end != null) {
            if (endInclusive) {
                return end.compareTo(name) >= 0;
            } else {
                return end.compareTo(name) > 0;
            }
        }

        return true;
    }

    @Nullable
    public String getBegin() {
        return begin;
    }

    @Nullable
    public String getEnd() {
        return end;
    }

    public boolean isBeginInclusive() {
        return beginInclusive;
    }

    public boolean isEndInclusive() {
        return endInclusive;
    }

    public boolean isAnyInclusive() {
        return beginInclusive || endInclusive;
    }

    public boolean isAll() {
        return begin == null && end == null;
    }

    @Override
    public String toString() {
        return "GenericRange{" +
            "begin=" + begin +
            ", end=" + end +
            ", beginInclusive=" + beginInclusive +
            ", endInclusive=" + endInclusive +
            '}';
    }
}
