package ru.yandex.kikimr.client.kv;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.Iterables;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.kikimr.client.KikimrGrpcTransport;
import ru.yandex.kikimr.client.KikimrTransport;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.solomon.kikimr.LocalKikimr;


/**
 * @author Stepan Koltsov
 */
public class KikimrKvClientTest {

    @ClassRule
    public static LocalKikimr kikimr = new LocalKikimr();
    private KikimrKvClient kikimrKvClient;
    private KikimrKvClientSync kikimrKvClientSync;

    @Before
    public void before() {
        KikimrTransport transport = new KikimrGrpcTransport(Collections.singletonList(kikimr.getGrpcEndpoint()));
        kikimrKvClient = new KikimrKvClientImpl(transport);
        kikimrKvClientSync = new KikimrKvClientSync(kikimrKvClient);
    }

    @After
    public void after() {
        if (kikimrKvClientSync != null) {
            kikimrKvClientSync.close();
        }
    }

    private long createKvTablet() {
        String path = "/local/" + UUID.randomUUID().toString();
        kikimrKvClientSync.createKvTablesWithSchemeShard(path, 1);
        long[] tabletIds = kikimrKvClientSync.resolveKvTablesWithSchemeShard(path);
        return tabletIds[0];
    }

    @Test
    public void createKvTablets() {
        kikimrKvClientSync.createKvTablesWithSchemeShard("/local/kv1", 1);
        kikimrKvClientSync.createKvTablesWithSchemeShard("/local/kv2", 2);
        {
            long[] tabletIds = kikimrKvClientSync.resolveKvTablesWithSchemeShard("/local/kv1");
            Assert.assertEquals(1, tabletIds.length);
        }
        {
            long[] tabletIds = kikimrKvClientSync.resolveKvTablesWithSchemeShard("/local/kv2");
            Assert.assertEquals(2, tabletIds.length);
        }
    }

    @Test
    public void localEnumerateTablets() {
        String path = "/local/localEnumerateTablets";
        kikimrKvClientSync.createKvTablesWithSchemeShard(path, 3);
        long[] tablets = kikimrKvClientSync.resolveKvTablesWithSchemeShard(path);

        outer:
        for (int i = 0; ; ++i) {
            if (i == 100) {
                Assert.fail();
            }

            long[] found = kikimrKvClientSync.findTabletsOnLocalhost();
            for (long tablet : tablets) {
                if (Arrays.stream(found).noneMatch(t -> t == tablet)) {
                    ThreadUtils.sleep(10, TimeUnit.MILLISECONDS);
                    continue outer;
                }
            }

            break;
        }
    }

    @Test
    public void incrementGeneration() {
        long tabletId = createKvTablet();

        long gen = kikimrKvClientSync.incrementGeneration(tabletId);
        long gen1 = kikimrKvClientSync.incrementGeneration(tabletId);
        Assert.assertEquals(gen1, gen + 1);
    }

    @Test
    public void rejectWrongGeneration() {
        long tabletId = createKvTablet();
        long gen = kikimrKvClientSync.incrementGeneration(tabletId);

        kikimrKvClientSync.writeDefault(
            tabletId, gen, "aabb", "cc".getBytes(StandardCharsets.UTF_8));

        kikimrKvClientSync.incrementGeneration(tabletId);

        try {
            kikimrKvClient.write(
                tabletId, gen, "aabb", "dd".getBytes(StandardCharsets.UTF_8),
                MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN, KikimrKvClient.Write.defaultPriority, 0).join();

            Assert.fail("expected exception not thrown");
        } catch (CompletionException e) {
            if (!(e.getCause() instanceof KikimrKvGenerationChangedRuntimeException)) {
                Assert.fail("expected KikimrKvGenerationChangedRuntimeException cause");
            }
            // expected
        }
    }

    private KvTabletIdAndGen commonInitTablet() {
        long tabletId = createKvTablet();
        long gen = kikimrKvClientSync.incrementGeneration(tabletId);
        kikimrKvClientSync.deleteAll(tabletId, gen);
        return new KvTabletIdAndGen(tabletId, gen);
    }

    @Test
    public void readNonExistentKey() {
        KvTabletIdAndGen tablet = commonInitTablet();

        Optional<byte[]> read = kikimrKvClientSync.readData(tablet.getTabletId(), tablet.getGen(), "nonexistent", 0, -1);
        Assert.assertFalse(read.isPresent());
    }

    @Test
    public void writeRead() {
        KvTabletIdAndGen tablet = commonInitTablet();

        byte[] written = "ccdd".getBytes(StandardCharsets.UTF_8);
        kikimrKvClientSync.writeDefault(tablet.getTabletId(), tablet.getGen(), "aabb", written);

        byte[] read = kikimrKvClientSync.readData(tablet.getTabletId(), tablet.getGen(), "aabb", 0, -1).get();

        Assert.assertArrayEquals(written, read);
    }

    @Test
    public void readRangeBeyond() {
        KvTabletIdAndGen tablet = commonInitTablet();

        byte[] written = "ccdd".getBytes(StandardCharsets.US_ASCII);
        kikimrKvClientSync.writeDefault(tablet.getTabletId(), tablet.getGen(), "aabb", written);

        // test what happens if reading data beyond EOF

        Assert.assertArrayEquals(
            written,
            kikimrKvClientSync.readDataSome(tablet.getTabletId(), tablet.getGen(), "aabb", 0, 20));

        Assert.assertArrayEquals(
            new byte[0],
            kikimrKvClientSync.readDataSome(tablet.getTabletId(), tablet.getGen(), "aabb", 20, 30));
    }

    private void readDataLargeSomeImpl(KvTabletIdAndGen t, int secondFileSize) {
        Random2 r = new Random2(10);
        byte[] b1 = r.nextBytes(10_000);
        byte[] b2 = r.nextBytes(secondFileSize);
        byte[] expected = ByteBufUtil.getBytes(Unpooled.wrappedBuffer(b1, b2));

        kikimrKvClientSync.writeDefault(t.getTabletId(), t.getGen(), "t1", b1);
        kikimrKvClientSync.writeDefault(t.getTabletId(), t.getGen(), "t2", b2);

        kikimrKvClient.concatAndDeleteOriginals(t.getTabletId(), t.getGen(), List.of("t1", "t2"), "a", 0).join();

        Assert.assertArrayEquals(
            expected,
                kikimrKvClient.readDataLarge(t.getTabletId(), t.getGen(), "a", 0, MsgbusKv.TKeyValueRequest.EPriority.REALTIME).join());
        Assert.assertArrayEquals(
            Arrays.copyOfRange(expected, 2, expected.length - 3),
                kikimrKvClient.readDataLarge(t.getTabletId(), t.getGen(), 2, expected.length - 5, "a", 0, MsgbusKv.TKeyValueRequest.EPriority.REALTIME).join());
    }

    @Test
    public void readDataLargeSome() {
        kikimrKvClientSync.kikimrKvClient.setDoNotExceedFileSizeForTest(10_000);

        KvTabletIdAndGen t = commonInitTablet();

        readDataLargeSomeImpl(t, 10_000);
        readDataLargeSomeImpl(t, 10_000 - 2);
    }

    @Test
    public void writeSsd() {
        KvTabletIdAndGen tablet = commonInitTablet();

        byte[] written = "ccdd".getBytes(StandardCharsets.UTF_8);
        kikimrKvClientSync.writeDefault(tablet.getTabletId(), tablet.getGen(), "aabb", written);

        byte[] read = kikimrKvClientSync.readData(tablet.getTabletId(), tablet.getGen(), "aabb", 0, -1).get();

        Assert.assertArrayEquals(written, read);
    }

    @Test
    public void readMulti() {
        KvTabletIdAndGen tablet = commonInitTablet();

        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "aabb", "ab".getBytes(StandardCharsets.UTF_8));
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "ccdd", "cd".getBytes(StandardCharsets.UTF_8));

        {
            byte[][] r = kikimrKvClientSync.readDataMulti(tablet.getTabletId(), tablet.getGen(),
                new KvChunkAddress[]{ new KvChunkAddress("aabb", 1, 1), new KvChunkAddress("ccdd") });

            byte[][] expected = new byte[][]{
                "b".getBytes(StandardCharsets.UTF_8),
                "cd".getBytes(StandardCharsets.UTF_8)
            };

            Assert.assertArrayEquals(expected, r);
        }

        {
            byte[][] r = kikimrKvClientSync.readDataMulti(tablet.getTabletId(), tablet.getGen(),
                new KvChunkAddress[] { new KvChunkAddress("aabb", 1, 1), new KvChunkAddress("nonexistent") });

            byte[][] expected = new byte[][]{
                "b".getBytes(StandardCharsets.UTF_8),
                "".getBytes(StandardCharsets.UTF_8),
            };

            Assert.assertArrayEquals(expected, r);
        }

    }

    @Test
    public void readRange() {
        KvTabletIdAndGen tablet = commonInitTablet();

        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "a.004", "ddd".getBytes(StandardCharsets.UTF_8));
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "a.001", "aaa".getBytes(StandardCharsets.UTF_8));
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "a.003", "ccc".getBytes(StandardCharsets.UTF_8));
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "a.002", "bbb".getBytes(StandardCharsets.UTF_8));

        {
            NameRange range = NameRange.inclusiveExclusive("a.001", "a.003");
            List<KikimrKvClient.KvEntryWithStats> entries = Arrays.asList(kikimrKvClientSync
                .readRange(tablet.getTabletId(), tablet.getGen(), range, false)
                .getEntriesAll())
                .stream()
                .map(e -> e.withUnixtime(1))
                .collect(Collectors.toList());

            List<KikimrKvClient.KvEntryWithStats> expected = List.of(
                new KikimrKvClient.KvEntryWithStats("a.001", "".getBytes(StandardCharsets.UTF_8), 3, 1),
                new KikimrKvClient.KvEntryWithStats("a.002", "".getBytes(StandardCharsets.UTF_8), 3, 1)
            );

            Assert.assertEquals(expected, entries);
        }

        {
            NameRange range = NameRange.inclusiveExclusive("a.001", "a.003");
            List<KikimrKvClient.KvEntryWithStats> entries = Arrays.asList(kikimrKvClientSync
                .readRange(tablet.getTabletId(), tablet.getGen(), range, true)
                .getEntriesAll())
                .stream()
                .map(e -> e.withUnixtime(1))
                .collect(Collectors.toList());

            List<KikimrKvClient.KvEntryWithStats> expected = List.of(
                new KikimrKvClient.KvEntryWithStats("a.001", "aaa".getBytes(StandardCharsets.UTF_8), 3, 1),
                new KikimrKvClient.KvEntryWithStats("a.002", "bbb".getBytes(StandardCharsets.UTF_8), 3, 1)
            );

            Assert.assertEquals(expected, entries);
        }
    }

    @Test
    public void rename() {
        KvTabletIdAndGen tablet = commonInitTablet();

        byte[] written = "vvv".getBytes(StandardCharsets.UTF_8);
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "aaa", written);

        kikimrKvClientSync.rename(tablet.getTabletId(), tablet.getGen(), "aaa", "bbb");

        Assert.assertFalse(kikimrKvClientSync.readDataExists(tablet.getTabletId(), tablet.getGen(), "aaa"));

        byte[] read = kikimrKvClientSync.readData(tablet.getTabletId(), tablet.getGen(), "bbb", 0, -1).get();

        Assert.assertArrayEquals(written, read);
    }

    @Test
    public void renameNonexistent() {
        KvTabletIdAndGen tablet = commonInitTablet();

        try {
            kikimrKvClientSync.rename(tablet.getTabletId(), tablet.getGen(), "aaa", "bbb");
            Assert.fail();
        } catch (Exception e) {
            e.printStackTrace();
            // expected
        }
    }

    @Test
    public void delete() {
        KvTabletIdAndGen tablet = commonInitTablet();

        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "a001", "xxx".getBytes(CharsetUtils.UTF8_CHARSET));
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "a002", "yyy".getBytes(CharsetUtils.UTF8_CHARSET));
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "a003", "zzz".getBytes(CharsetUtils.UTF8_CHARSET));
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "a004", "www".getBytes(CharsetUtils.UTF8_CHARSET));

        kikimrKvClientSync.deleteRange(tablet.getTabletId(), tablet.getGen(), NameRange.inclusive("a002", "a003"));

        Assert.assertTrue(kikimrKvClientSync.readDataExists(tablet.getTabletId(), tablet.getGen(), "a001"));
        Assert.assertFalse(kikimrKvClientSync.readDataExists(tablet.getTabletId(), tablet.getGen(), "a002"));
        Assert.assertFalse(kikimrKvClientSync.readDataExists(tablet.getTabletId(), tablet.getGen(), "a003"));
        Assert.assertTrue(kikimrKvClientSync.readDataExists(tablet.getTabletId(), tablet.getGen(), "a004"));
    }

    @Test
    public void deleteNonexistent() {
        KvTabletIdAndGen tablet = commonInitTablet();

        // not an error
        kikimrKvClient.deleteRange(tablet.getTabletId(), tablet.getGen(), NameRange.single("nonex"), 0).join();
    }

    @Test
    public void readHundreds() {
        KvTabletIdAndGen tablet = commonInitTablet();

        List<CompletableFuture<Void>> futures = IntStream.range(0, 10).mapToObj(i -> {
            return kikimrKvClient.write(
                tablet.getTabletId(), tablet.getGen(),
                String.format("a%04d", i),
                ("x" + i).getBytes(StandardCharsets.UTF_8),
                MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN,
                KikimrKvClient.Write.defaultPriority, 0);
        }).collect(Collectors.toList());

        CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new)).join();

        NameRange range = NameRange.inclusive("a0002", "a0005");
        List<KikimrKvClient.KvEntryWithStats> entries = Arrays.stream(
            kikimrKvClientSync
                .readRange(tablet.getTabletId(), tablet.getGen(), range, true)
                .getEntriesAll())
            .map(e -> e.withUnixtime(1))
            .collect(Collectors.toList());

        List<KikimrKvClient.KvEntryWithStats> expected = IntStream.rangeClosed(2, 5)
            .mapToObj(i -> {
                return new KikimrKvClient.KvEntryWithStats(String.format("a%04d", i), ("x" + i).getBytes(StandardCharsets.UTF_8), ("x" + i).getBytes(StandardCharsets.UTF_8).length, 1);
            })
            .collect(Collectors.toList());

        Assert.assertEquals(expected, entries);
    }

    @Test
    public void writeLarge() {
        KvTabletIdAndGen tablet = commonInitTablet();

        String fileName = "aaaa";

        Random r = new Random(17);
        byte[] buf = new byte[30_000_000];
        r.nextBytes(buf);

        try {
            kikimrKvClientSync.writeDefault(tablet.getTabletId(), tablet.getGen(), fileName, buf);
            Assert.fail("Call unexpectedly passed");
        } catch (IllegalArgumentException expected) {
            // too large
        }

        AtomicInteger idSupplier = new AtomicInteger();
        Supplier<String> chunkNameSupplier =  () -> fileName + ".temp." + idSupplier.getAndIncrement();

        kikimrKvClientSync.writeLarge(tablet.getTabletId(), tablet.getGen(), fileName, buf,
            chunkNameSupplier,
            MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN,
            MsgbusKv.TKeyValueRequest.EPriority.REALTIME
        );
        byte[] read = kikimrKvClientSync.readDataLarge(tablet.getTabletId(), tablet.getGen(), fileName);
        Assert.assertArrayEquals(buf, read);
    }

    @Test
    public void readMoreThanFitsMessage() {
        KvTabletIdAndGen tablet = commonInitTablet();

        for (int i = 0; i < 10; ++i) {
            System.out.println("write " + i);
            kikimrKvClientSync.writeDefault(
                tablet.getTabletId(), tablet.getGen(), "a" + i, array(8000, (byte) i));
        }

        {
            System.out.println("read overrun");
            KvReadRangeResult readRange = kikimrKvClientSync
                .readRange(tablet.getTabletId(), tablet.getGen(), NameRange.inclusive("a2", "a9"), true, 9000);
            Assert.assertTrue(readRange.isOverrun());
        }

        {
            System.out.println("read over limit");
            try {
                kikimrKvClientSync
                    .readRange(tablet.getTabletId(), tablet.getGen(), NameRange.inclusive("a2", "a9"), true, 7000);
                Assert.fail();
            } catch (Exception e) {
                // OK
            }
        }

        {
            System.out.println("read with iter");

            KvAsyncIterator<ArrayList<KikimrKvClient.KvEntryWithStats>> iter = kikimrKvClient.readRangeAllIter(
                tablet.getTabletId(), tablet.getGen(), NameRange.inclusive("a2", "a9"), 9000, 0);

            for (int i = 2; i < 10; ++i) {
                ArrayList<KikimrKvClient.KvEntryWithStats> e = iter.next().join().get();
                KikimrKvClient.KvEntryWithStats entry = Iterables.getOnlyElement(e.stream().collect(Collectors.toList()));
                Assert.assertEquals("a" + i, entry.getName());
                Assert.assertArrayEquals(array(8000, (byte) i), entry.getValue());
                iter.commit();
            }

            Assert.assertTrue(!iter.next().join().isPresent());
        }

        {
            System.out.println("read 3");

            KvReadRangeResult r = kikimrKvClientSync.readRange(
                tablet.getTabletId(), tablet.getGen(),
                NameRange.inclusive("a2", "a9"), true, 17000);

            Assert.assertTrue(r.isOverrun());
            Assert.assertEquals(2, r.getEntries().length);
            Assert.assertEquals("a2", r.getEntries()[0].getName());
            Assert.assertArrayEquals(array(8000, (byte) 2), r.getEntries()[0].getValue());
            Assert.assertEquals("a3", r.getEntries()[1].getName());
            Assert.assertArrayEquals(array(8000, (byte) 3), r.getEntries()[1].getValue());
        }
    }

    @Test
    public void cloneRange() {
        KvTabletIdAndGen tablet = commonInitTablet();

        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "aabb", new byte[] { 10, 20 });
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "aacc", new byte[] { 30 });
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(), "xxyy", new byte[] { 40, 50, 60 });

        kikimrKvClient.copyRange(tablet.getTabletId(), tablet.getGen(), NameRange.all(), "bb", "aa", 0).join();

        Assert.assertArrayEquals(new byte[] { 10, 20 }, kikimrKvClientSync.readData(tablet.getTabletId(), tablet.getGen(), "bbbb", 0, -1).get());
        Assert.assertArrayEquals(new byte[] { 30 }, kikimrKvClientSync.readData(tablet.getTabletId(), tablet.getGen(), "bbcc", 0, -1).get());
    }

    @Test
    public void concat() {
        KvTabletIdAndGen tablet = commonInitTablet();

        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(),
            "aa", "xx".getBytes(StandardCharsets.US_ASCII));
        kikimrKvClientSync.writeDefault(
            tablet.getTabletId(), tablet.getGen(),
            "bb", "yy".getBytes(StandardCharsets.US_ASCII));

        kikimrKvClient.concatAndDeleteOriginals(
            tablet.getTabletId(), tablet.getGen(),
            List.of("aa", "bb"), "cc", 0).join();

        Assert.assertEquals(List.of("cc"),
            kikimrKvClientSync.readRangeNames(tablet.getTabletId(), tablet.getGen())
                .stream().map(KikimrKvClient.KvEntryStats::getName).collect(Collectors.toList()));

        Assert.assertArrayEquals(
            "xxyy".getBytes(StandardCharsets.US_ASCII),
            kikimrKvClientSync.readDataSome(tablet.getTabletId(), tablet.getGen(), "cc"));
    }

    private static byte[] array(int size, byte value) {
        byte[] array = new byte[size];
        Arrays.fill(array, value);
        return array;
    }
}
