package ru.yandex.kikimr.client.kv.noderesolver;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.function.Supplier;

import com.google.common.net.HostAndPort;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.kikimr.client.discovery.StaticNodeDiscovery;
import ru.yandex.kikimr.client.kv.transport.NodeStubFactoryImpl;

/**
 * @author senyasdr
 */
@SuppressWarnings("UnstableApiUsage")
public class KikmirV2NodeResolverTest {


    private StaticNodeDiscovery discovery;
    private KikmirV2NodeResolverImpl nodeResolver;

    @Before
    public void before() {
        discovery = new StaticNodeDiscovery(new HashMap<>() {{
            put(1, HostAndPort.fromString("https://127.0.0.1:80"));
            put(2, HostAndPort.fromString("https://127.0.0.1:81"));
            put(3, HostAndPort.fromString("https://127.0.0.1:82"));
        }});
        nodeResolver = new KikmirV2NodeResolverImpl(discovery, ForkJoinPool.commonPool(), new NodeStubFactoryImpl());
    }

    @Test
    public void testGetNodeIdByPathAndPartitionIdWhenNodeIsPresent() {
        // given
        nodeResolver.updateNodeId("testpath", 1, 1);
        // when
        var testNode = nodeResolver.getNodeIdByPathAndPartitionIdOrRandom("testpath", 1);

        // then
        Assert.assertEquals(HostAndPort.fromString("https://127.0.0.1:80"), testNode.hostAndPort());
        Assert.assertEquals(1, testNode.nodeId());
    }

    @Test
    public void testGetNodeIdByPathAndPartitionIdWhenNodeIsNotPresent() {
        // when
        Assert.assertTrue(checkIsResultRandom(() -> nodeResolver.getNodeIdByPathAndPartitionIdOrRandom("testpath", 1), 3));
    }

    @Test
    public void testUpdateNodeIdWhenNodeIsPresentInHostMap() {
        // given
        Assert.assertTrue(checkIsResultRandom(() -> nodeResolver.getNodeIdByPathAndPartitionIdOrRandom("testpath", 1), 3));

        // when
        nodeResolver.updateNodeId("testpath", 3, 1);

        // then
        var testNode = nodeResolver.getNodeIdByPathAndPartitionIdOrRandom("testpath", 3);

        Assert.assertEquals(HostAndPort.fromString("https://127.0.0.1:80"), testNode.hostAndPort());
        Assert.assertEquals(1, testNode.nodeId());
    }

    @Test(timeout = 100)
    public void testUpdateNodeIdWhenNodeIsNotPresentInHostMap() {
        // given
        Assert.assertTrue(checkIsResultRandom(() -> nodeResolver.getNodeIdByPathAndPartitionIdOrRandom("testpath", 1), 3));

        // when
        CompletableFuture<Void> doneFuture = nodeResolver.waitForUpdateFuture();
        nodeResolver.updateNodeId("testpath", 3, 4);

        //then
        doneFuture.join();
    }

    @Test
    public void testInvalidateTabletAndGetNew() {
        // given
        nodeResolver.updateNodeId("testpath", 1, 1);
        var testNode = nodeResolver.getNodeIdByPathAndPartitionIdOrRandom("testpath", 1);
        Assert.assertEquals(HostAndPort.fromString("https://127.0.0.1:80"), testNode.hostAndPort());
        Assert.assertEquals(1, testNode.nodeId());

        // when
        nodeResolver.invalidateNodeAndGetNew("testpath", 1);

        // then
        Assert.assertTrue(checkIsResultRandom(() -> nodeResolver.getNodeIdByPathAndPartitionIdOrRandom("testpath", 1), 2, 250));
    }

    @Test(timeout = 300)
    public void testActualizingNodes() {
        // giver
        CompletableFuture<Void> doneFuture = nodeResolver.waitForUpdateFuture();
        nodeResolver.updateNodeId("testpath", 1, 1);
        nodeResolver.updateNodeId("testpath", 2, 2);
        nodeResolver.updateNodeId("testpath", 3, 3);

        // when
        discovery.updateAddresses(
                new HashMap<>() {{
                    put(2, HostAndPort.fromString("https://127.0.0.1:83"));
                    put(3, HostAndPort.fromString("https://127.0.0.1:82"));
                    put(4, HostAndPort.fromString("https://127.0.0.1:81"));
                    put(5, HostAndPort.fromString("https://127.0.0.1:84"));
                }}
        );
        doneFuture.join();

        // then
        Assert.assertTrue(nodeResolver.getNodeIdByPathAndPartitionId("testpath", 1).isClosed());
        Assert.assertTrue(nodeResolver.getNodeIdByPathAndPartitionId("testpath", 2).isClosed());
        Assert.assertTrue(checkIsResultRandom(() -> nodeResolver.getNodeIdByPathAndPartitionIdOrRandom("testpath", 1), 4));
        Assert.assertTrue(checkIsResultRandom(() -> nodeResolver.getNodeIdByPathAndPartitionIdOrRandom("testpath", 2), 4));

        Assert.assertEquals(HostAndPort.fromString("https://127.0.0.1:82"), nodeResolver.getNodeIdByPathAndPartitionIdOrRandom("testpath", 3).hostAndPort());

        Assert.assertNull(nodeResolver.getNodeIdByPathAndPartitionId("testpath", 1));
        Assert.assertNull(nodeResolver.getNodeIdByPathAndPartitionId("testpath", 2));
    }

    public static final int delta = 150;
    public static final int iterations = 1000;

    private <T> boolean checkIsResultRandom(Supplier<T> checker, int size, int delta) {
        Map<T, Integer> distribution = new HashMap<>();
        for (int i = 0; i < iterations; i++) {
            T res = checker.get();
            if (distribution.containsKey(res)) {
                Integer count = distribution.get(res);
                count++;
                distribution.put(res, count);
            } else {
                distribution.put(res, 1);
            }
        }
        return distribution.values().stream().allMatch(i ->
                i < (iterations / size) + delta && i > (iterations / size) - delta);
    }

    private <T> boolean checkIsResultRandom(Supplier<T> checker, int size) {
        return checkIsResultRandom(checker, size, delta);
    }
}
