package ru.yandex.solomon.labels;

import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;


/**
 * <pre>
 *    Benchmark                                             Mode  Cnt   Score    Error   Units
 *    SelectorsMatchingBenchmark.absentSelector            thrpt    5  46.659 ±  4.227  ops/us
 *    SelectorsMatchingBenchmark.anySelector               thrpt    5  39.422 ±  6.194  ops/us
 *    SelectorsMatchingBenchmark.exactSelector             thrpt    5  41.378 ± 10.501  ops/us
 *    SelectorsMatchingBenchmark.exactSelectorAsGlob       thrpt    5  37.216 ±  3.444  ops/us
 *    SelectorsMatchingBenchmark.globSelector              thrpt    5   9.097 ±  1.460  ops/us
 *    SelectorsMatchingBenchmark.multiGlobSelector         thrpt    5  20.968 ±  2.910  ops/us
 *    SelectorsMatchingBenchmark.notExactSelector          thrpt    5  33.822 ±  5.789  ops/us
 *    SelectorsMatchingBenchmark.notGlobSelector           thrpt    5  27.507 ±  3.319  ops/us
 *    SelectorsMatchingBenchmark.regexSelector             thrpt    5   0.914 ±  0.151  ops/us
 *    SelectorsMatchingBenchmark.regexWithCaptureSelector  thrpt    5   0.816 ±  0.131  ops/us
 * </pre>
 *
 * @author Oleg Baryshnikov
 */
@Fork(value = 1)
@Warmup(iterations = 5, time = 500, timeUnit = TimeUnit.MILLISECONDS)
@Measurement(iterations = 5, time = 1, timeUnit = TimeUnit.SECONDS)
@BenchmarkMode(Mode.Throughput)
@OutputTimeUnit(TimeUnit.MICROSECONDS)
@ParametersAreNonnullByDefault
public class SelectorsMatchingBenchmark {

    private static final Selectors ANY_SELECTOR = Selectors.parse("host=*");
    private static final Selectors ABSENT_SELECTOR = Selectors.parse("host=-");
    private static final Selectors EXACT_SELECTOR = Selectors.parse("host==cluster");
    private static final Selectors EXACT_GLOB_SELECTOR = Selectors.parse("host=cluster");
    private static final Selectors NOT_EXACT_SELECTOR = Selectors.parse("host!==cluster");
    private static final Selectors GLOB_SELECTOR = Selectors.parse("host=solomon-*-sas");
    private static final Selectors NOT_GLOB_SELECTOR = Selectors.parse("host=solomon-*");
    private static final Selectors MULTI_GLOB_SELECTOR = Selectors.parse("host=cluster|solomon-*");
    private static final Selectors REGEX_SELECTOR = Selectors.parse("host=~solomon-.*-sas");
    private static final Selectors REGEX_WITH_CAPTURE_SELECTOR = Selectors.parse("host=~solomon-(?P<actual_name>.*)-sas");

    private static final Labels labels = Labels.of(
            "cluster", "prestable",
            "service", "frontend",
            "host", "solomon-pre-man-13");

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(SelectorsMatchingBenchmark.class.getName())
                .detectJvmArgs()
                .jvmArgs("-Xmx3g", "-Xms3g")
                .build();

        new Runner(opt).run();
    }

    @Benchmark
    public boolean anySelector() {
        return ANY_SELECTOR.match(labels);
    }

    @Benchmark
    public boolean absentSelector() {
        return ABSENT_SELECTOR.match(labels);
    }

    @Benchmark
    public boolean exactSelector() {
        return EXACT_SELECTOR.match(labels);
    }

    @Benchmark
    public boolean exactSelectorAsGlob() {
        return EXACT_GLOB_SELECTOR.match(labels);
    }

    @Benchmark
    public boolean notExactSelector() {
        return NOT_EXACT_SELECTOR.match(labels);
    }

    @Benchmark
    public boolean globSelector() {
        return GLOB_SELECTOR.match(labels);
    }

    @Benchmark
    public boolean notGlobSelector() {
        return NOT_GLOB_SELECTOR.match(labels);
    }

    @Benchmark
    public boolean multiGlobSelector() {
        return MULTI_GLOB_SELECTOR.match(labels);
    }

    @Benchmark
    public boolean regexSelector() {
        return REGEX_SELECTOR.match(labels);
    }

    @Benchmark
    public boolean regexWithCaptureSelector() {
        return REGEX_WITH_CAPTURE_SELECTOR.match(labels);
    }
}
