package ru.yandex.solomon.labels;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.solomon.util.collection.array.ArrayBuilder;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class InterpolatedString {

    private final String[] literals;
    private final String[] variables;

    public InterpolatedString(String[] literals, String[] variables) {
        if (literals.length != variables.length + 1) {
            throw new IllegalArgumentException("incorrect literals and variables lengths");
        }
        this.literals = literals;
        this.variables = variables;
    }

    public List<String> variables() {
        return List.of(variables);
    }

    private int variableCount() {
        return variables.length;
    }

    public Optional<String> constant() {
        if (literals.length == 1) {
            return Optional.of(literals[0]);
        }
        return Optional.empty();
    }

    @Nonnull
    public String eval(Function<String, String> resolver) {
        // optimization
        Optional<String> constant = constant();
        if (constant.isPresent()) {
            return constant.get();
        }

        StringBuilder sb = new StringBuilder();
        sb.append(literals[0]);
        for (int i = 0; i < variables.length; ++i) {
            String r = resolver.apply(variables[i]);
            if (r == null) {
                throw new IllegalArgumentException("failed to resolve variable: " + variables[i]);
            }
            sb.append(r);
            sb.append(literals[i + 1]);
        }
        return sb.toString();
    }

    @Nonnull
    public InterpolatedString partialEval(Function<String, Optional<String>> resolver) {
        ArrayBuilder<String> literals = new ArrayBuilder<>(new String[1]);
        ArrayBuilder<String> variables = new ArrayBuilder<>(Cf.StringArray.emptyArray());
        literals.add(this.literals[0]);
        for (int i = 0; i < this.variables.length; ++i) {
            Optional<String> value = resolver.apply(this.variables[i]);
            if (value.isPresent()) {
                literals.array[literals.size - 1] += value.get() + this.literals[i + 1];
            } else {
                variables.add(this.variables[i]);
                literals.add(this.literals[i + 1]);
            }
        }
        return new InterpolatedString(literals.buildUnsafe(), variables.buildUnsafe());
    }

    public static boolean isInterpolatedString(String source) {
        return source.contains("{{");
    }

    public static InterpolatedString parse(String pattern) {
        ArrayBuilder<String> literals = new ArrayBuilder<>(new String[1]);
        ArrayBuilder<String> variables = new ArrayBuilder<>(Cf.StringArray.emptyArray());

        int pos = 0;
        for (;;) {
            int nextStart = pattern.indexOf("{{", pos);
            if (nextStart < 0) {
                literals.add(pattern.substring(pos));
                break;
            }

            int nextEnd = pattern.indexOf("}}", nextStart);
            if (nextEnd < 0) {
                throw new IllegalArgumentException("wrong pattern: " + pattern);
            }

            literals.add(pattern.substring(pos, nextStart));
            String variable = pattern.substring(nextStart + "{{".length(), nextEnd);
            if (variable.isEmpty()) {
                throw new IllegalArgumentException("empty variable");
            }
            variables.add(variable);
            pos = nextEnd + "}}".length();
        }

        return new InterpolatedString(literals.buildUnsafe(), variables.buildUnsafe());
    }

    @Override
    public String toString() {
        return format();
    }

    @Nonnull
    public String format() {
        return eval(name -> "{{" + name + "}}");
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        InterpolatedString that = (InterpolatedString) o;

        // Probably incorrect - comparing Object[] arrays with Arrays.equals
        if (!Arrays.equals(literals, that.literals)) return false;
        // Probably incorrect - comparing Object[] arrays with Arrays.equals
        return Arrays.equals(variables, that.variables);

    }

    @Override
    public int hashCode() {
        int result = Arrays.hashCode(literals);
        result = 31 * result + Arrays.hashCode(variables);
        return result;
    }
}
