package ru.yandex.solomon.labels;

import java.util.Arrays;
import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.solomon.labels.selector.LabelSelector;

/**
 * @author Stepan Koltsov
 */
public class LabelValueSelector {

    public enum MatchOp {
        PRESENT(LabelSelector.MatchOp.PRESENT),
        GLOB_POSITIVE(LabelSelector.MatchOp.GLOB_POSITIVE),
        GLOB_NEGATIVE(LabelSelector.MatchOp.GLOB_NEGATIVE),
        ;

        private final LabelSelector.MatchOp labelMatchOp;

        MatchOp(LabelSelector.MatchOp labelMatchOp) {
            this.labelMatchOp = labelMatchOp;
        }

        public LabelSelector.MatchOp getLabelMatchOp() {
            return labelMatchOp;
        }
    }

    @Nullable
    private final String[] patterns;
    private final boolean positive;

    public LabelValueSelector(@Nullable String[] patterns, boolean positive) {
        this.patterns = patterns;
        this.positive = positive;

        if (!positive) {
            if (patterns == null) {
                throw new IllegalArgumentException("incorrect label value selector");
            }
        }
    }

    public static LabelValueSelector any() {
        return new LabelValueSelector(null, true);
    }

    public static LabelValueSelector exactPositive(String value) {
        return new LabelValueSelector(new String[] { value }, true);
    }

    public MatchOp matchOp() {
        if (positive) {
            if (patterns != null) {
                return MatchOp.GLOB_POSITIVE;
            } else {
                return MatchOp.PRESENT;
            }
        } else {
            if (patterns != null) {
                return MatchOp.GLOB_NEGATIVE;
            } else {
                throw new IllegalStateException("incorrect label value selector");
            }
        }
    }

    public boolean matches(@Nonnull String value) {
        if (this.patterns == null) {
            return positive;
        } else {
            boolean anyMatch = false;
            for (String pattern : this.patterns) {
                if (LabelValueGlob.glob(pattern, value)) {
                    anyMatch = true;
                    break;
                }
            }
            return anyMatch == positive;
        }
    }

    public boolean anyMatch(Collection<String> values) {
        for (String value : values) {
            if (matches(value)) {
                return true;
            }
        }
        return false;
    }

    public boolean isWildcardAll() {
        return patterns == null;
    }

    public boolean isAll() {
        return positive && isWildcardAll();
    }

    public boolean isExactPositive() {
        return positive
                && patterns != null
                && patterns.length == 1
                && !LabelValueGlob.isGlob(patterns[0]);
    }

    @Nullable
    public String[] getValueRaw() {
        return patterns;
    }

    public boolean isPositive() {
        return positive;
    }

    @Nonnull
    public static LabelValueSelector parse(@Nonnull String value) {
        String[] patterns;
        boolean positive;
        if (value.equals(LabelValues.ANY)) {
            patterns = null;
            positive = true;
        } else if (value.startsWith("!")) {
            patterns = value.substring("!".length()).split("\\|");
            positive = false;
        } else {
            patterns = value.split("\\|");
            positive = true;
        }
        return new LabelValueSelector(patterns, positive);
    }

    @Override
    public String toString() {
        return "LabelValueSelector{" +
            "patterns=" + Arrays.toString(patterns) +
            ", positive=" + positive +
            '}';
    }
}
