package ru.yandex.solomon.labels;

import java.util.Collections;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.util.Quoter;
import ru.yandex.solomon.util.parser.ParserSupport;

/**
 * Analogue of LabelsString
 * We must use new format: key1="value1", key2="value2" (quotes are optional) in new code
 * and support old LabelsString format: key1=value1&key2=value2
 *
 * @see ru.yandex.solomon.labels.LabelsString
 *
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class LabelsFormat {

    private static final char LABEL_DELIMITER = ',';
    private static final char KEY_VALUE_DELIMITER = '=';

    private static final Set<Character> LABEL_KEY_STOP_SYMBOLS =
        Collections.singleton(KEY_VALUE_DELIMITER);

    private static final Set<Character> LABEL_VALUE_STOP_SYMBOLS =
        Collections.singleton(LABEL_DELIMITER);

    public static Labels parse(String text) {
        String trimedText = text.trim();

        if (trimedText.isEmpty()) {
            return Labels.empty();
        }

        ParserSupport parser = new ParserSupport(trimedText);

        LabelsBuilder labelsBuilder = Labels.builder();
        consumeLabel(parser, labelsBuilder);

        while (parser.hasNext()) {
            parser.consume(LABEL_DELIMITER);
            consumeLabel(parser, labelsBuilder);
        }

        return labelsBuilder.build();
    }

    private static void consumeLabel(ParserSupport parser, LabelsBuilder labelsBuilder) {
        parser.consumeWhitespaces();

        String labelKey = Quoter.consumeQuotedString(parser, LABEL_KEY_STOP_SYMBOLS);

        parser.consumeWhitespaces();

        parser.consume(KEY_VALUE_DELIMITER);

        parser.consumeWhitespaces();

        String labelValue = Quoter.consumeQuotedString(parser, LABEL_VALUE_STOP_SYMBOLS);

        parser.consumeWhitespaces();

        labelsBuilder.add(labelKey, labelValue);
    }

    public static String format(Labels labels) {
        if (labels.isEmpty()) {
            return "";
        }

        StringBuilder sb = new StringBuilder(labels.size() * 10);
        labels.forEach(l -> {
            sb.append(l.getKey()).append(KEY_VALUE_DELIMITER).append(Quoter.doubleQuote(l.getValue()));
            sb.append(LABEL_DELIMITER);
            sb.append(' ');
        });
        sb.setLength(sb.length() - 2);
        return sb.toString();
    }

    public static String format(Label label) {
        return label.getKey() + KEY_VALUE_DELIMITER + Quoter.doubleQuote(label.getValue());
    }
}
