package ru.yandex.solomon.labels;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.labels.query.SelectorsException;


/**
 * @author Sergey Polovko
 */
public final class LabelsString {
    private LabelsString() {}

    private static final char LABELS_DELIMITER = '&';
    private static final char KEY_VALUE_DELIMITER = '=';

    private static final LabelAllocator allocator = new InterningLabelAllocator();


    public static Labels parse(String[] parts) {
        return parse(parts, 0);
    }

    public static Labels parse(String[] parts, int begin) {
        LabelsBuilder builder = Labels.builder(parts.length - begin);
        for (int i = begin; i < parts.length; i++) {
            builder.add(parseLabel(parts[i]));
        }
        return builder.build();
    }

    public static Labels parse(String str) {
        return parse(str, LABELS_DELIMITER);
    }

    public static Labels parse(String str, char delimiter) {
        if (str.isEmpty()) {
            return Labels.empty();
        }

        int index = str.indexOf(delimiter);
        if (index == -1) {
            return Labels.of(parseLabel(str));
        }

        LabelsBuilder builder = Labels.builder(5);
        int begin = 0;
        while (true) {
            builder.add(parseLabel(str.substring(begin, index)));
            begin = index + 1;

            index = str.indexOf(delimiter, begin);
            if (index == -1) {
                builder.add(parseLabel(str.substring(begin)));
                break;
            }
        }
        return builder.build();
    }

    public static Label parseLabel(String str) {
        int index = str.indexOf(KEY_VALUE_DELIMITER);
        if (index == -1) {
            throw new SelectorsException("wrong label format: " + str);
        }
        String key = str.substring(0, index);
        String value = str.substring(index + 1);
        return allocator.alloc(key, value);
    }

    public static String format(Labels labels) {
        return format(labels, LABELS_DELIMITER);
    }

    public static String format(Labels labels, char delimiter) {
        if (labels.isEmpty()) {
            return "";
        }

        StringBuilder sb = new StringBuilder(labels.size() * 10);
        labels.forEach(l -> {
            sb.append(l.getKey()).append(KEY_VALUE_DELIMITER).append(l.getValue());
            sb.append(delimiter);
        });
        sb.setLength(sb.length() - 1); // remove last delimiter
        return sb.toString();
    }

    public static String format(Label label) {
        return label.getKey() + KEY_VALUE_DELIMITER + label.getValue();
    }
}
