package ru.yandex.solomon.labels.intern;

import java.lang.reflect.Field;
import java.util.Map;

import com.google.common.collect.Interner;
import com.google.common.collect.Interners;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.util.ExceptionUtils;


/**
 * @author Sergey Polovko
 */
public class InterningLabelAllocator implements LabelAllocator, MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(InterningLabelAllocator.class);

    // object header + 2 refs + 1 int
    private static final long WeakKeyStrongValueEntry_SELF_SIZE = MemoryCounter.OBJECT_HEADER_SIZE +
        2 * MemoryCounter.OBJECT_POINTER_SIZE
        + Integer.BYTES;

    // object header = 2 refs
    private static final long StringLabel_SELF_SIZE = MemoryCounter.OBJECT_HEADER_SIZE +
        2 * MemoryCounter.OBJECT_POINTER_SIZE;

    private final Interner<String> stringInterner;
    private final Map<String, ?> stringInternerMap;
    private final Interner<Label> labelInterner;
    private final Map<?, ?> labelInternerMap;

    private volatile long stringInternerSize = -1;
    private volatile long stringInternerByteSize;

    public InterningLabelAllocator() {
        this.stringInterner = Interners.newWeakInterner();
        this.labelInterner = Interners.newWeakInterner();

        stringInternerMap = getMapField(stringInterner);
        labelInternerMap = getMapField(stringInterner);
    }

    @Override
    public Label alloc(String key, String value) {
        String keyInterned = internKey(key);
        String valueInterned = interValue(value);
        Label label = Labels.allocator.alloc(keyInterned, valueInterned);
        return labelInterner.intern(label);
    }

    public String internKey(String key) {
        return stringInterner.intern(key);
    }

    public String interValue(String value) {
        return stringInterner.intern(value);
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        if (stringInternerSize == stringInternerMap.size()) {
            size += stringInternerByteSize;
        } else {
            int length = 0;
            long stringInternerByteSize = 0;
            for (String s : stringInternerMap.keySet()) {
                stringInternerByteSize += MemoryCounter.stringSize(s) + WeakKeyStrongValueEntry_SELF_SIZE;
                length++;
            }

            this.stringInternerSize = length;
            this.stringInternerByteSize = stringInternerByteSize;
            size += stringInternerByteSize;
        }

        size += labelInternerMap.size() * (StringLabel_SELF_SIZE + WeakKeyStrongValueEntry_SELF_SIZE);
        return size;
    }

    private static <K, V> Map<K, V> getMapField(Interner<?> interner) {
        try {
            Class<? extends Interner> aClass = interner.getClass();
            Field mapField = aClass.getDeclaredField("map");
            mapField.setAccessible(true);

            //noinspection unchecked
            return (Map<K, V>) mapField.get(interner);
        } catch (Exception e) {
            ExceptionUtils.uncaughtException(e);
            throw new Error(e);
        }
    }
}
