package ru.yandex.solomon.labels.query;

import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.labels.query.matchers.GlobMatcher;


/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class GlobSelector implements Selector {

    private final String key;
    private final String value;
    private final GlobMatcher matcher;

    GlobSelector(String key, String value) {
        this.key = key;
        this.value = value;

        // It's necessary to prevent wrong host="!cluster" format in selectors
        if (value.startsWith("!")) {
            throw new SelectorsException("Glob selector value cannot start with \"!\". Valid syntax is label!=\"value\"");
        }

        this.matcher = GlobMatcher.from(value);
    }

    @Override
    public String getKey() {
        return key;
    }

    @Override
    public String getValue() {
        return value;
    }

    public GlobMatcher getMatcher() {
        return matcher;
    }

    @Override
    public SelectorType getType() {
        return SelectorType.GLOB;
    }

    @Override
    public boolean match(@Nullable String value) {
        return matcher.matches(value);
    }

    @Override
    public <T> void forEachMatchedKey(Map<String, T> values, Consumer<T> fn) {
        matcher.forEachMatchedKey(values, fn);
    }

    @Override
    public Selector withKey(String key) {
        return new GlobSelector(key, value);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof GlobSelector)) {
            return false;
        }

        GlobSelector that = (GlobSelector) o;

        if (!key.equals(that.key)) {
            return false;
        }
        if (!value.equals(that.value)) {
            return false;
        }
        return matcher.equals(that.matcher);
    }

    @Override
    public int hashCode() {
        int result = key.hashCode();
        result = 31 * result + value.hashCode();
        result = 31 * result + matcher.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "GlobSelector{" +
            "key='" + key + '\'' +
            ", value='" + value + '\'' +
            ", matcher=" + matcher +
            '}';
    }
}
