package ru.yandex.solomon.labels.query;

import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.labels.LabelValues;


/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public interface Selector {

    /**
     * Selector key
     */
    String getKey();

    /**
     * Selector value
     */
    String getValue();

    /**
     * Selector type
     */
    SelectorType getType();

    default boolean isExact() {
        if (this instanceof ExactSelector) {
            return true;
        }
        if (this instanceof GlobSelector) {
            return ((GlobSelector) this).getMatcher().isExact();
        }
        return false;
    }

    /**
     * Check that selector match the existing label value
     *
     * @param value label value
     * @return true if selector match the label value
     */
    boolean match(@Nullable String value);

    /**
     * Call consumer for each associated value which key is matched by this selector.
     * Depending on type of the selector this method uses the best strategy to find value(s) in map.
     *
     * @param values map with key-value pairs
     * @param fn consumer
     * @param <T> type of values
     */
    default <T> void forEachMatchedKey(Map<String, T> values, Consumer<T> fn) {
        for (Map.Entry<String, T> e : values.entrySet()) {
            if (match(e.getKey())) {
                fn.accept(e.getValue());
            }
        }
    }

    /**
     * Create same selector with another key
     *
     * @param key new key
     * @return copy of selector with new key
     */
    Selector withKey(String key);

    /**
     * Create selector using key, value and type
     */
    static Selector of(String key, String value, SelectorType type) {
        return type.create(key, value);
    }

    /**
     * Create absent selector, e.g.: host=-
     */
    static Selector absent(String key) {
        return new AbsentSelector(key);
    }

    /**
     * Create any selector, e.g.: host=*
     */
    static Selector any(String key) {
        return new AnySelector(key);
    }

    /**
     * Create exact selector, e.g.: host==cluster
     */
    static Selector exact(String key, String value) {
       return new ExactSelector(key, value);
    }

    /**
     * Create negative exact selector, e.g.: host!==cluster
     */
    static Selector notExact(String key, String value) {
       return new NotSelector(Selector.exact(key, value), SelectorType.NOT_EXACT);
    }

    /**
     * Create glob selector, e.g.: host=solomon-*
     */
    static Selector glob(String key, String value) {
        return new GlobSelector(key, value);
    }

    /**
     * Create negative glob selector: host!=solomon-*
     */
    static Selector notGlob(String key, String value) {
        return new NotSelector(glob(key, value), SelectorType.NOT_GLOB);
    }

    /**
     * Create glob, not glob, any or absent selector in depends on selector value.
     * For example:
     * <ul>
     *   <li>host=- - absent selector</li>
     *   <li>host=* - any selector</li>
     *   <li>host=cluster - glob selector</li>
     * </ul>
     */
    static Selector extendedGlob(String key, String value) {
        if (LabelValues.ANY.equals(value)) {
            return any(key);
        }
        if (LabelValues.ABSENT.equals(value)) {
            return absent(key);
        }
        return glob(key, value);
    }

    /**
     * Create not glob, absent or any selector in depends on selector value automatically.
     * For example:
     * <ul>
     *   <li>host!=- - any selector</li>
     *   <li>host!=* - absent selector</li>
     *   <li>host!=cluster - not glob selector</li>
     * </ul>
     */
    static Selector extendedNotGlob(String key, String value) {
        if (LabelValues.ANY.equals(value)) {
            return absent(key);
        }
        if (LabelValues.ABSENT.equals(value)) {
            return any(key);
        }
        return notGlob(key, value);
    }

    /**
     * Create regex selector, e.g.: host=~\w+
     */
    static Selector regex(String key, String value) {
        return new RegexSelector(key, value);
    }

    /**
     * Create not regex selector, e.g.: host!~\w+
     */
    static Selector notRegex(String key, String value) {
        return new NotSelector(new RegexSelector(key, value), SelectorType.NOT_REGEX);
    }
}
