package ru.yandex.solomon.labels.query;

import java.util.function.BiFunction;

/**
 * @author Oleg Baryshnikov
 */
public enum SelectorType {
    GLOB(0, "=", Selector::extendedGlob),
    NOT_GLOB(1, "!=", Selector::extendedNotGlob),
    EXACT(2, "==", Selector::exact),
    NOT_EXACT(3, "!==", Selector::notExact),
    REGEX(4, "=~", Selector::regex),
    NOT_REGEX(5, "!~", Selector::notRegex),
    ANY(6, "=", (key, value) -> Selector.any(key)),
    ABSENT(7, "=", (key, value) -> Selector.absent(key)),
    ;

    private final int number;
    private final String operator;
    private final BiFunction<String, String, Selector> factory;

    SelectorType(int number, String operator, BiFunction<String, String, Selector> factory) {
        this.number = number;
        this.operator = operator;
        this.factory = factory;
    }

    public static SelectorType forNumber(int number) {
        for (SelectorType type : values()) {
            if (type.getNumber() == number) {
                return type;
            }
        }

        throw new SelectorsException("Unknown number: " + number);
    }

    public static SelectorType forOperator(String operator) {
        if ("=".equals(operator)) {
            return GLOB;
        }

        for (SelectorType type : values()) {
            if (type.getOperator().equals(operator)) {
                return type;
            }
        }

        throw new SelectorsException("Unknown operator: " + operator);
    }

    public String getOperator() {
        return operator;
    }

    public int getNumber() {
        return number;
    }

    public Selector create(String key, String value) {
        return factory.apply(key, value);
    }
}
