package ru.yandex.solomon.labels.query;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.shard.ShardKey;

/**
 * Common manipulations with selectors
 *
 * @author Oleg Baryshnikov
 */
@Immutable
@ParametersAreNonnullByDefault
public abstract class Selectors extends SelectorsStaticPart implements Iterable<Selector> {
    // - size & index access -

    public abstract String getNameSelector();

    public abstract int size();

    public abstract boolean isEmpty();

    public abstract Selector at(int index);

    // - iteration -

    @Override
    @Nonnull
    public abstract Iterator<Selector> iterator();

    public abstract Stream<Selector> stream();

    // - find -

    public abstract boolean hasKey(String key);

    @Nullable
    public abstract Selector findByKey(String key);

    // - transformations -

    public abstract SelectorsBuilder toBuilder();

    public abstract List<Selector> toList();

    // - match -

    /**
     * Check that selector.match labels (new implementation)
     *
     * @param labels labels in new format
     * @return true if selector.match labels, false otherwise
     */
    public boolean match(Labels labels) {
        int size = size();
        for (int index = 0; index < size; index++) {
            Selector selector = at(index);
            Label label = labels.findByKey(selector.getKey());

            boolean result = label == null
                    ? selector.match(null)
                    : selector.match(label.getValue());

            if (!result) {
                return false;
            }
        }

        return true;
    }

    public boolean match(ShardKey key) {
        int size = size();
        for (int index = 0; index < size; index++) {
            Selector selector = at(index);
            final boolean result;
            switch (selector.getKey()) {
                case LabelKeys.PROJECT:
                    result = selector.match(key.getProject());
                    break;
                case LabelKeys.CLUSTER:
                    result = selector.match(key.getCluster());
                    break;
                case LabelKeys.SERVICE:
                    result = selector.match(key.getService());
                    break;
                default:
                    result = selector.match(null);
            }

            if (!result) {
                return false;
            }
        }

        return true;
    }

    public boolean match(Map<String, String> labels) {
        int size = size();
        for (int index = 0; index < size; index++) {
            Selector selector = at(index);
            var value = labels.get(selector.getKey());

            boolean result = value == null
                    ? selector.match(null)
                    : selector.match(value);

            if (!result) {
                return false;
            }
        }

        return true;
    }

    // - helper functions -

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getNameSelector());
        sb.append('{');
        for (int index = 0; index < size(); index++) {
            Selector selector = at(index);
            String k = selector.getKey();
            String v = selector.getValue();
            String op = selector.getType().getOperator();

            sb.append(k).append(op).append('\'').append(v).append('\'').append(',').append(' ');
        }
        if (size() > 0) {
            sb.setLength(sb.length() - 2);
        }
        sb.append('}');
        return sb.toString();
    }
}
