package ru.yandex.solomon.labels.query;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

/**
 * @author Vladimir Gordiychuk
 */
public class SelectorsBuilder {
    private String nameSelector;
    private final List<Selector> selectors;

    public SelectorsBuilder(String nameSelector, List<Selector> selectors) {
        this.nameSelector = nameSelector;
        this.selectors = new ArrayList<>(selectors);
    }

    public SelectorsBuilder(String nameSelector, int capacity) {
        this.nameSelector = nameSelector;
        this.selectors = new ArrayList<>(capacity);
    }

    public Selector at(int index) {
        return selectors.get(index);
    }

    public boolean hasKey(String key) {
        for (Selector selector : selectors) {
            if (selector.getKey().equals(key)) {
                return true;
            }
        }
        return false;
    }

    @Nullable
    public Selector findByKey(String key) {
        for (Selector selector : selectors) {
            if (selector.getKey().equals(key)) {
                return selector;
            }
        }
        return null;
    }

    public SelectorsBuilder setNameSelector(String nameSelector) {
        this.nameSelector = nameSelector;
        return this;
    }

    public SelectorsBuilder add(String key, String value) {
        return add(SelectorType.GLOB.create(key, value));
    }

    public SelectorsBuilder add(Selector selector) {
        selectors.add(selector);
        return this;
    }

    public SelectorsBuilder addAll(Collection<Selector> selectors) {
        this.selectors.addAll(selectors);
        return this;
    }

    public SelectorsBuilder addAll(Selectors selectors) {
        for (Selector selector : selectors) {
            this.selectors.add(selector);
        }
        return this;
    }

    public SelectorsBuilder combine(SelectorsBuilder builder) {
        this.selectors.addAll(builder.selectors);
        return this;
    }

    public SelectorsBuilder addOverride(String key, String value) {
        return addOverride(SelectorType.GLOB.create(key, value));
    }

    public SelectorsBuilder addOverride(Selector selector) {
        selectors.removeIf(s -> s.getKey().equals(selector.getKey()));
        selectors.add(selector);
        return this;
    }

    public SelectorsBuilder addOverride(Selectors selectors) {
        Set<String> keys = selectors.stream()
                .map(Selector::getKey)
                .collect(Collectors.toSet());

        this.selectors.removeIf(selector -> keys.contains(selector.getKey()));
        for (Selector selector : selectors) {
            this.selectors.add(selector);
        }
        return this;
    }

    public SelectorsBuilder remove(Selector selector) {
        selectors.remove(selector);
        return this;
    }

    public SelectorsBuilder remove(String key) {
        selectors.removeIf(selector -> selector.getKey().equals(key));
        return this;
    }

    public SelectorsBuilder remove(int index) {
        selectors.remove(index);
        return this;
    }

    public Selectors build() {
        if (selectors.size() == 0) {
            if (nameSelector.isEmpty()) {
                return Selectors0.INSTANCE;
            }

            return new Selectors0(nameSelector);
        }

        return new SelectorsArray(nameSelector, selectors.toArray(new Selector[0]));
    }
}
