package ru.yandex.solomon.labels.query;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.monlib.metrics.labels.validate.StrictValidator;
import ru.yandex.solomon.util.Quoter;
import ru.yandex.solomon.util.parser.ParserSupport;

/**
 * @author Oleg Baryshnikov
 */
public final class SelectorsFormat {
    private static final String[] SUPPORTED_OPERATORS =
            new String[]{"!==", "!=", "!~", "==", "=~", "="};

    private static final char SELECTOR_DELIMITER = ',';

    public static final Pattern METRIC_NAME_PATTERN = Pattern.compile("(?i)[a-z_][a-z0-9_.]*");

    private static final HashSet<Character> FORBIDDEN_SYMBOLS =
            new HashSet<>(Arrays.asList('"', '\'', '{', '}', '=', '!', '&', SELECTOR_DELIMITER));

    private SelectorsFormat() {
    }

    /**
     * Parses selectors text query
     * @param text selectors query in text format
     */
    public static Selectors parse(String text) {
        ParserSupport parserSupport = new ParserSupport(text.trim());
        try {
            return consumeSelectors(parserSupport, false);
        } catch (SelectorsException e) {
            throw e;
        } catch (Exception e) {
            throw new SelectorsException(e);
        }
    }

    public static Selector parseSelector(String text) {
        ParserSupport parserSupport = new ParserSupport(text.trim());
        try {
            return consumeSelector(parserSupport);
        } catch (SelectorsException e) {
            throw e;
        } catch (Exception e) {
            throw new SelectorsException(e);
        }
    }

    public static Selectors consumeSelectors(ParserSupport parserSupport, boolean mustBeWrapped) {
        if (!parserSupport.hasNext()) {
            return Selectors.of();
        }

        SelectorsBuilder selectors = Selectors.builder();

        parserSupport.consumeWhitespaces();

        String nameSelector = "";
        boolean quoted = false;

        try {
            ParserSupport copiedParser = parserSupport.copy();
            nameSelector = consumeQuotedNameSelector(copiedParser);
            copiedParser.consume('{');
            copiedParser.copyTo(parserSupport);
            quoted = true;
        } catch (Exception e) {
            nameSelector = "";
            if (parserSupport.lookaheadIs('{')) {
                parserSupport.consume('{');
                quoted = true;
            }
        }

        if (mustBeWrapped && !quoted) {
            throw new SelectorsException("selectors must be wrapped by curly brackets");
        }

        selectors.setNameSelector(nameSelector);

        if (quoted && parserSupport.lookaheadIs('}')) {
            parserSupport.consume('}');
            return selectors.build();
        }

        selectors.add(consumeSelector(parserSupport));

        while (parserSupport.hasNext() && !parserSupport.lookaheadIs('}')) {
            parserSupport.consume(SELECTOR_DELIMITER);
            selectors.add(consumeSelector(parserSupport));
        }

        if (quoted) {
            parserSupport.consume('}');
        }

        return selectors.build();
    }

    public static String format(Selectors selectors) {
        String nameSelector = selectors.getNameSelector();

        String nameSelectorPrefix = formatNameSelector(nameSelector);

        String selectorsPart = formatSelectors(selectors.stream());

        return nameSelectorPrefix + '{' + selectorsPart + '}';
    }

    private static String formatNameSelector(String nameSelector) {
        final String nameSelectorPrefix;

        if (nameSelector.isEmpty()) {
            nameSelectorPrefix = "";
        } else if (METRIC_NAME_PATTERN.matcher(nameSelector).matches()) {
            nameSelectorPrefix = nameSelector;
        } else {
            nameSelectorPrefix = Quoter.doubleQuote(nameSelector);
        }

        return nameSelectorPrefix;
    }

    private static String formatSelectors(Stream<Selector> selectors) {
        return selectors
                .map(SelectorsFormat::format)
                .collect(Collectors.joining(", "));
    }

    public static String format(Selector selector) {
        String formattedKey = StrictValidator.validateKey(selector.getKey()) == null ? selector.getKey() : Quoter.singleQuote(selector.getKey());
        String formattedValue = Quoter.singleQuote(selector.getValue());
        return formattedKey + selector.getType().getOperator() + formattedValue;
    }

    private static String consumeQuotedNameSelector(ParserSupport parserSupport) {
        final String nameSelector;

        if (parserSupport.lookaheadIs(c -> c == '"' || c == '\'')) {
           nameSelector = Quoter.consumeQuotedString(parserSupport, Collections.emptySet());
        } else {
            nameSelector = consumeNameSelector(parserSupport);
        }

        parserSupport.consumeWhitespaces();

        return nameSelector;
    }

    private static String consumeNameSelector(ParserSupport parserSupport) {
        Optional<String> metricNameOpt = parserSupport.consumeOptional(METRIC_NAME_PATTERN);
        return metricNameOpt.orElse("");
    }

    private static Selector consumeSelector(ParserSupport parserSupport) {
        parserSupport.consumeWhitespaces();

        String labelKey = consumeQuotedString(parserSupport);

        parserSupport.consumeWhitespaces();

        Optional<String> operatorOpt = parserSupport.consumeOptionalOneOf(SUPPORTED_OPERATORS);
        if (operatorOpt.isEmpty()) {
            throw new SelectorsException("unknown selector operator: " + parserSupport.consumeRest());
        }
        String operator = operatorOpt.get();

        parserSupport.consumeWhitespaces();

        String labelValue = consumeQuotedString(parserSupport);

        parserSupport.consumeWhitespaces();

        return constructSelector(labelKey, labelValue, operator);
    }

    private static Selector constructSelector(
        String key,
        String value,
        String operator)
    {
        return SelectorType.forOperator(operator).create(key, value);
    }

    private static String consumeQuotedString(ParserSupport parserSupport) {
        return Quoter.consumeQuotedString(parserSupport, FORBIDDEN_SYMBOLS);
    }
}
