package ru.yandex.solomon.labels.query;

import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.labels.validate.LabelsValidator;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelValueGlob;

/**
 * @author Vladimir Gordiychuk
 */
public final class SelectorsValidator {
    private SelectorsValidator() {
    }

    public static boolean isValid(SelectorType type, String key, String value) {
        return isKeyValid(key) && isValueValid(type, value)
            && isValidProjectLabel(type, key, value);
    }

    private static boolean isValidProjectLabel(SelectorType type, String key, String value) {
        if (LabelKeys.PROJECT.equals(key)) {
            return type == SelectorType.EXACT
                || type == SelectorType.GLOB && !(LabelValueGlob.isGlob(value) || value.contains("|"));
        }

        return true;
    }

    static boolean isKeyValid(String key) {
        return LabelsValidator.isKeyValid(key);
    }

    static boolean isValueValid(SelectorType type, String value) {
        switch (type) {
            case EXACT:
            case NOT_EXACT:
                return LabelsValidator.isValueValid(value);
            case REGEX:
            case NOT_REGEX:
                try {
                    Pattern pattern = Pattern.compile(value);
                    return true;
                } catch (PatternSyntaxException e) {
                    return false;
                }
            case ABSENT:
            case ANY:
                return true;
            case NOT_GLOB:
            case GLOB:
                return isGlobValid(value);
            default:
                throw new SelectorsException("Unsupported selector type: " + type);
        }
    }

    private static boolean isGlobValid(String value) {
        for (String glob : StringUtils.split(value, "|")) {
            if ("".equals(glob)) {
                return false;
            }

            int from = 0;
            for (int index = 0; index < glob.length(); index++) {
                char c = glob.charAt(index);
                switch (c) {
                    case '*':
                    case '?': {
                        if (from != index && !LabelsValidator.isValueValid(glob.substring(from, index))) {
                            return false;
                        }

                        from = index + 1;
                        break;
                    }
                }
            }

            if (from < glob.length() && !LabelsValidator.isValueValid(glob.substring(from))) {
                return false;
            }
        }
        return true;
    }
}
