package ru.yandex.solomon.labels.query;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.shard.ShardKey;


/**
 * @author Oleg Baryshnikov
 */
public class ShardSelectors {
    /**
     * Get shard key from selectors
     */
    @Nullable
    public static ShardKey getShardKeyOrNull(Selectors selectors) {
        Selector project = selectors.findByKey(LabelKeys.PROJECT);
        if (project == null || !project.isExact()) {
            return null;
        }

        Selector cluster = selectors.findByKey(LabelKeys.CLUSTER);
        if (cluster == null || !cluster.isExact()) {
            return null;
        }

        Selector service = selectors.findByKey(LabelKeys.SERVICE);
        if (service == null || !service.isExact()) {
            return null;
        }

        return new ShardKey(project.getValue(), cluster.getValue(), service.getValue());
    }

    /**
     * Return shard part of selectors
     *
     * @param selectors selectors
     * @return new selectors with shard selectors only
     */
    public static Selectors onlyShardKey(Selectors selectors) {
        List<Selector> selectorsOfShardKeyPart = selectors.stream()
            .filter(selector -> LabelKeys.isShardKeyPart(selector.getKey()))
            .collect(Collectors.toList());

        return Selectors.of(selectorsOfShardKeyPart);
    }

    public static Split split(Selectors selectors) {
        SelectorsBuilder shardSelector = Selectors.builder(3);
        SelectorsBuilder metricSelector = Selectors.builder(selectors.getNameSelector(), selectors.size());
        for (var selector : selectors) {
            if (LabelKeys.isShardKeyPart(selector.getKey())) {
                shardSelector.add(selector);
            } else {
                metricSelector.add(selector);
            }
        }

        return new Split(shardSelector.build(), metricSelector.build());
    }

    /**
     * Return inside shard part of selectors
     *
     * @param selectors selectors
     * @return new selectors without shard selectors only
     */
    public static Selectors withoutShardKey(Selectors selectors) {
        return selectors.toBuilder()
            .remove(LabelKeys.PROJECT)
            .remove(LabelKeys.CLUSTER)
            .remove(LabelKeys.SERVICE)
            .build();
    }

    /**
     * Check that all shard selectors have exact values
     * Need to prevent pagination for cross-shard metabase requests
     *
     * @param selectors selectors
     * @return true if all shard selectors have exact values, e.g.: project=junk, cluster=foo, service=bar
     */
    public static boolean isSingleShard(Selectors selectors) {
        if (!selectors.getNameSelector().isEmpty()) {
            return false;
        }

        Selector project = selectors.findByKey(LabelKeys.PROJECT);
        if (project == null || !project.isExact()) {
            return false;
        }

        Selector cluster = selectors.findByKey(LabelKeys.CLUSTER);
        if (cluster == null || !cluster.isExact()) {
            return false;
        }

        Selector service = selectors.findByKey(LabelKeys.SERVICE);
        return service != null && service.isExact();
    }

    /**
     * Concat (project, cluster, service) and after shard selectors without common keys check
     *
     * @param shardKey (project, cluster, service) triple
     * @param selectorsAfterShard selectors after shard: host=cluster, sensor=some_sensor_name
     * @return concatenated selectors
     */
    public static Selectors concat(ShardKey shardKey, Selectors selectorsAfterShard) {
        return Selectors.builder(selectorsAfterShard.getNameSelector())
            .add(LabelKeys.PROJECT, shardKey.getProject())
            .add(LabelKeys.CLUSTER, shardKey.getCluster())
            .add(LabelKeys.SERVICE, shardKey.getService())
            .addAll(selectorsAfterShard)
            .build();
    }

    public record Split(Selectors shardSelector, Selectors metricsSelector) {
    }
}
