package ru.yandex.solomon.labels.query.matchers;

import java.util.Arrays;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.labels.LabelValueGlob;
import ru.yandex.solomon.labels.LabelValues;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
class MultiGlobMatcher implements GlobMatcher {

    private final GlobMatcher[] matchers;

    MultiGlobMatcher(String pattern) {
        String[] subPatterns = StringUtils.split(pattern, '|');

        this.matchers = Arrays.stream(subPatterns)
            .map(subPattern -> {
                if (LabelValues.ANY.equals(subPattern)) {
                    return AnyGlobMatcher.INSTANCE;
                }

                if (LabelValues.ABSENT.equals(subPattern)) {
                    return AbsentGlobMatcher.INSTANCE;
                }

                if (LabelValueGlob.isGlob(subPattern)) {
                    return new SingleGlobMatcher(subPattern);
                }

                return new ExactGlobMatcher(subPattern);
            })
            .toArray(GlobMatcher[]::new);
    }

    @Override
    public boolean matches(@Nullable String value) {
        for (GlobMatcher matcher : matchers) {
            if (matcher.matches(value)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public <T> void forEachMatchedKey(Map<String, T> values, Consumer<T> fn) {
        for (GlobMatcher matcher : matchers) {
            matcher.forEachMatchedKey(values, fn);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof MultiGlobMatcher)) {
            return false;
        }

        MultiGlobMatcher that = (MultiGlobMatcher) o;

        // Probably incorrect - comparing Object[] arrays with Arrays.equals
        return Arrays.equals(matchers, that.matchers);
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(matchers);
    }

    @Override
    public String toString() {
        return "MultiGlobMatcher{" +
            "matchers=" + Arrays.toString(matchers) +
            '}';
    }
}
