package ru.yandex.solomon.labels.selector;

import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.solomon.labels.LabelValueSelector;
import ru.yandex.solomon.labels.LabelValues;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class LabelSelector {

    public enum MatchOp {
        /** Label value is equal to specified value */
        GLOB_POSITIVE,
        /** Label is present and value is not equal to specified value */
        GLOB_NEGATIVE,
        /** Label is present */
        PRESENT,
        /** Label is absent */
        ABSENT,
    }

    private static final Pattern legacy1 = Pattern.compile("!(.*)=\\*");
    private static final Pattern parserPattern = Pattern.compile("([^!=]+)(!?)=(.+)", Pattern.DOTALL);
    @Nonnull
    private final String name;
    @Nonnull // null means any
    private final String value;
    private final boolean valueIsAlreadyParsed;

    public LabelSelector(String name, String value) {
        this(name, value, false);
    }

    public LabelSelector(String name, String value, boolean valueIsAlreadyParsed) {
        this.name = name;
        this.value = value;
        this.valueIsAlreadyParsed = valueIsAlreadyParsed;
    }

    public LabelSelectorParsed toParsed() {
        return LabelSelectorParsed.parse(name, value, valueIsAlreadyParsed);
    }

    @Nonnull
    public String getName() {
        return name;
    }

    public boolean isAll() {
        return valueSelector().isPresent() && valueSelector().get().isAll();
    }

    public boolean isPositive() {
        return toParsed().isPositive();
    }

    public boolean isVariable() {
        switch (matchOp()) {
            case ABSENT:
                return false;
            case PRESENT:
            case GLOB_NEGATIVE:
                return true;
            case GLOB_POSITIVE:
                return !valueSelector().get().isExactPositive();
            default:
                throw new IllegalStateException("unknown match operator: " + matchOp());
        }
    }

    public MatchOp matchOp() {
        return toParsed().getMatchOp();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        LabelSelector selector = (LabelSelector) o;

        if (!name.equals(selector.name)) return false;
        if (!value.equals(selector.value)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + value.hashCode();
        return result;
    }

    @Nonnull
    public String getValueFormatted() {
        return value;
    }

    @Nonnull
    public Optional<LabelValueSelector> valueSelector() {
        return toParsed().valueSelector();
    }

    public String format() {
        return name + "=" + value;
    }

    public String formatUrlEncoded() {
        return UrlUtils.urlEncode(name) + "=" + UrlUtils.urlEncode(value);
    }

    public String formatEscaped() {
        return name + "=" + LabelSelectorSet.escapeValue(value);
    }

    @Nonnull
    public static LabelSelector parse(String name, String value) {
        return LabelSelectorParsed.parse(name, value, false).toSelector();
    }

    @Nonnull
    public static LabelSelector parse(String string) {
        {
            // legacy syntax
            Matcher matcher = legacy1.matcher(string);
            if (matcher.matches()) {
                return parse(matcher.group(1), LabelValues.ABSENT);
            }
        }

        // != is legacy syntax, not sure if it is used anywhere
        Matcher matcher = parserPattern.matcher(string);
        if (!matcher.matches()) {
            throw new RuntimeException("cannot parse string as selector: " + string);
        }
        String name = matcher.group(1);
        String value = matcher.group(3);
        if (!matcher.group(2).isEmpty()) {
            return parse(name, "!" + value);
        }
        return parse(name, value);
    }

    @Override
    public String toString() {
        return format();
    }

}
