package ru.yandex.solomon.labels.selector;

import java.util.Arrays;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.LabelValueGlob;
import ru.yandex.solomon.labels.LabelValueSelector;
import ru.yandex.solomon.labels.LabelValues;
import ru.yandex.solomon.labels.query.SelectorsException;


/**
 * @author Stepan Koltsov
 */
public class LabelSelectorParsed {
    @Nonnull
    private final String name;
    @Nullable
    private final String[] value;
    @Nonnull
    private final LabelSelector.MatchOp matchOp;

    public LabelSelectorParsed(@Nonnull String name, @Nullable String[] value, @Nonnull LabelSelector.MatchOp matchOp) {
        this.name = name;
        this.value = value;
        this.matchOp = matchOp;
        if ((matchOp == LabelSelector.MatchOp.PRESENT
            || matchOp == LabelSelector.MatchOp.ABSENT)
            && value != null) {
            throw new SelectorsException("value is required for selector: " + name);
        }
    }

    public LabelSelectorParsed(@Nonnull String name, LabelValueSelector valueSelector) {
        this(name, valueSelector.getValueRaw(), valueSelector.matchOp().getLabelMatchOp());
    }

    public static LabelSelectorParsed parse(String name, String value, boolean valueIsAlreadyParsed) {
        if (valueIsAlreadyParsed) {
            return new LabelSelectorParsed(name, new String[]{value}, LabelSelector.MatchOp.GLOB_POSITIVE);
        }
        if (value.equals(LabelValues.ABSENT)) {
            return new LabelSelectorParsed(name, null, LabelSelector.MatchOp.ABSENT);
        } else {
            LabelValueSelector valueSelector = LabelValueSelector.parse(value);
            return new LabelSelectorParsed(name, valueSelector);
        }
    }

    public Optional<LabelValueSelector> valueSelector() {
        switch (matchOp) {
        case GLOB_POSITIVE:
            return Optional.of(new LabelValueSelector(value, true));
        case PRESENT:
            return Optional.of(new LabelValueSelector(null, true));
        case GLOB_NEGATIVE:
            return Optional.of(new LabelValueSelector(value, false));
        case ABSENT:
            return Optional.empty();
        default:
            throw new SelectorsException("unknown match operator: " + matchOp);
        }
    }

    public LabelSelector toSelector() {
        return new LabelSelector(name, formatValue());
    }

    @Nonnull
    public String getName() {
        return name;
    }

    @Nullable
    public String[] getValue() {
        return value;
    }

    @Nonnull
    public LabelSelector.MatchOp getMatchOp() {
        return matchOp;
    }

    public String formatValue() {
        switch (matchOp) {
            case GLOB_POSITIVE: return Stream.of(value).collect(Collectors.joining("|"));
            case GLOB_NEGATIVE: return "!" + Stream.of(value).collect(Collectors.joining("|"));
            case PRESENT: return LabelValues.ANY;
            case ABSENT: return LabelValues.ABSENT;
            default: throw new IllegalStateException("unknown match operator: " + matchOp);
        }
    }

    public boolean isPositive() {
        switch (matchOp) {
            case GLOB_POSITIVE:
            case PRESENT:
                return true;
            case GLOB_NEGATIVE:
            case ABSENT:
                return false;
        }
        throw new IllegalStateException("unknown match operator: " + matchOp);
    }

    @Nullable
    public Label asExactLabel() {
        return isFixedString() ? Labels.allocator.alloc(name, value[0]) : null;
    }

    public boolean isFixedString() {
        return matchOp == LabelSelector.MatchOp.GLOB_POSITIVE
                && value.length == 1
                && !LabelValueGlob.isGlob(value[0])
                ;
    }

    private boolean runValue(String arg) {
        if (this.value == null) {
            throw new IllegalStateException("empty value");
        }

        for (String val : this.value) {
            if (LabelValueGlob.isGlob(val) && LabelValueGlob.glob(val, arg)) {
                return true;
            } else if (val.equals(arg)) {
                return true;
            }
        }
        return false;
    }

    private boolean matchesAbsentValue() {
        if (this.value == null) {
            return true;
        }

        for (String subValue : this.value) {
            if (LabelValues.ABSENT.equals(subValue)) {
                return true;
            }
        }

        return false;
    }

    public boolean matches(Labels labels) {
        Label label = labels.findByKey(name);
        return matchValue(label == null ? null : label.getValue());
    }

    public boolean matchValue(@Nullable String value) {
        switch (matchOp) {
            case ABSENT: return value == null;
            case PRESENT: return value != null;
            case GLOB_POSITIVE: return value == null && matchesAbsentValue() || value != null && runValue(value);
            case GLOB_NEGATIVE: return value != null && !runValue(value);
            default: throw new IllegalStateException("unknown match operator: " + matchOp);
        }
    }

    public String format() {
        return name + "=" + formatValue();
    }

    @Override
    public String toString() {
        return format();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        LabelSelectorParsed that = (LabelSelectorParsed) o;

        if (matchOp != that.matchOp) return false;
        if (!name.equals(that.name)) return false;
        if (!Arrays.equals(value, that.value)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + (value != null ? Arrays.hashCode(value) : 0);
        result = 31 * result + matchOp.hashCode();
        return result;
    }
}
