package ru.yandex.solomon.labels.selector;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelValues;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsBuilder;
import ru.yandex.solomon.labels.shard.ShardKey;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class LabelSelectorSet {
    public static final LabelSelectorSet empty = new LabelSelectorSet();

    private static final Pattern labelSepPattern = Pattern.compile("[&;]");

    private final List<LabelSelector> selectors;

    private LabelSelectorSet() {
        this(Collections.emptyList());
    }

    public LabelSelectorSet(List<LabelSelector> selectors) {
        this.selectors = selectors;
    }

    public LabelSelectorSet(Labels labels) {
        selectors = new ArrayList<>(labels.size());
        labels.forEach(l -> {
            selectors.add(new LabelSelector(l.getKey(), l.getValue()));
        });
    }

    public static LabelSelectorSet fromMap(Map<String, String> query) {
        return new LabelSelectorSet(query.entrySet().stream()
                    .map(e -> LabelSelector.parse(e.getKey(), e.getValue()))
                    .collect(Collectors.toList()));
    }

    public static LabelSelectorSet fromShardKey(ShardKey shardKey) {
        return LabelSelectorSet.fromMap(shardKey.toMap());
    }

    @Nonnull
    public static LabelSelectorSet single(String name, String valueExpr) {
        return new LabelSelectorSet(List.of(LabelSelector.parse(name, valueExpr)));
    }

    public List<LabelSelector> getSelectors() {
        return selectors;
    }

    public List<String> names() {
        return selectors.stream().map(LabelSelector::getName).collect(Collectors.toList());
    }

    public boolean isAll() {
        return selectors.isEmpty();
    }

    public boolean matchesAll(Labels labels) {
        return toParsed().matchesAll(labels);
    }

    @Nonnull
    public LabelSelectorSet and(LabelSelector selector) {
        return and(new LabelSelectorSet(Collections.singletonList(selector)));
    }

    @Nonnull
    public LabelSelectorSet and(LabelSelectorSet selectors) {
        List<LabelSelector> r = new ArrayList<>(this.selectors.size() + selectors.selectors.size());
        r.addAll(this.selectors);
        r.addAll(selectors.selectors);
        return new LabelSelectorSet(r);
    }

    private LabelSelectorSet filter(Predicate<LabelSelector> p) {
        return new LabelSelectorSet(selectors.stream().filter(p).collect(Collectors.toList()));
    }

    public LabelSelectorSet filterAll() {
        return filter(LabelSelector::isAll);
    }

    @Nonnull
    public LabelSelectorSet andOverride(LabelSelectorSet override) {
        HashSet<String> overrideNames = new HashSet<>(override.names());

        List<LabelSelector> r = new ArrayList<>();
        r.addAll(this.filter(s -> !overrideNames.contains(s.getName())).getSelectors());
        r.addAll(override.getSelectors());
        return new LabelSelectorSet(r);
    }

    public boolean containsForLabelName(String color) {
        return selectors.stream().anyMatch(s -> s.getName().equals(color));
    }

    @Nonnull
    public LabelSelectorSetParsed toParsed() {
        List<LabelSelectorParsed> parsed = selectors.stream()
            .map(LabelSelector::toParsed)
            .collect(Collectors.toList());
        return new LabelSelectorSetParsed(parsed);
    }

    /**
     * Convert old LabelSelectorSet selectors to new selectors
     *
     * @return new selectors list
     */
    public Selectors toNewSelectors() {
        SelectorsBuilder builder = Selectors.builder(selectors.size());
        for (LabelSelector legacy : selectors) {
            builder.add(toNewSelector(legacy));
        }
        return builder.build();
    }

    public Selector toNewSelector(LabelSelector legacy) {
        String name = legacy.getName();
        String value = legacy.getValueFormatted();
        if (LabelValues.ABSENT.equals(value)) {
            return Selector.absent(name);
        }
        if (LabelValues.ANY.equals(value)) {
            return Selector.any(name);
        }
        if (value.startsWith("!")) {
            return Selector.notGlob(name, value.substring(1));
        }
        return Selector.glob(name, value);
    }

    @Nonnull
    public String format() {
        return format(" ");
    }

    public String format(String separator) {
        return selectors.stream().map(LabelSelector::formatEscaped).collect(Collectors.joining(separator));
    }

    public static Stream<String> splitLabelsBySeparators(String s) {
        return labelSepPattern.splitAsStream(s);
    }

    public static LabelSelectorSet parseEscaped(String s) {
        List<LabelSelector> selectors = splitEscapedLabels(s)
            .map(LabelSelector::parse)
            .collect(Collectors.toList());
        return new LabelSelectorSet(selectors);
    }

    public static String escapeValue(String value) {
        StringBuilder result = new StringBuilder();

        boolean hasEscapedChars = false;
        result.append('"');
        for (char c : value.toCharArray()) {
            switch (c) {
                case '"':
                    hasEscapedChars = true;
                    result.append("\\\"");
                    break;
                case '\\':
                    hasEscapedChars = true;
                    result.append("\\\\");
                    break;
                case '\n':
                    hasEscapedChars = true;
                    result.append("\\n");
                    break;
                case ' ':
                case '&':
                case ';':
                    hasEscapedChars = true;
                    result.append(c);
                    break;
                default:
                    result.append(c);
            }
        }
        result.append('"');

        if (!hasEscapedChars) {
            return value;
        }
        return result.toString();
    }

    static Stream<String> splitEscapedLabels(String s) {
        List<String> selectors = new ArrayList<>();
        StringBuilder collector = new StringBuilder();

        boolean isEscaped = false;
        boolean isInQuotes = false;
        for (char c: s.toCharArray()) {
            if (!isEscaped && !isInQuotes && (c == '&' || c == ';')) {
                selectors.add(collector.toString());
                collector = new StringBuilder();
                continue;
            }
            if (!isEscaped) {
                switch (c) {
                    case '\\':
                        isEscaped = true;
                        break;
                    case '"':
                        isInQuotes = !isInQuotes;
                        break;
                    default:
                        collector.append(c);
                }
            } else {
                switch (c) {
                    case '&':
                    case ' ':   // leaved here for backward compatibility (already escaped space must be valid)
                    case ';':
                    case '\\':
                    case '"':
                        collector.append(c);
                        break;
                    case 'n':
                        collector.append('\n');
                        break;
                    default:
                        throw new RuntimeException("Wrong character is escaped: '" + c + "' in labels list " + s);
                }
                isEscaped = false;
            }
        }
        selectors.add(collector.toString());
        if (isEscaped) {
            throw new RuntimeException("Cannot parse labels set: non used escape symbol at the end found");
        }
        if (isInQuotes) {
            throw new RuntimeException("Cannot parse labels set: non-closed enquoted block found");
        }

        return selectors.stream().filter(l -> !l.isEmpty());
    }

    public void forEachSelector(BiConsumer<String, String> fn) {
        for (LabelSelector s : selectors) {
            fn.accept(s.getName(), s.getValueFormatted());
        }
    }

    @Nonnull
    public LabelSelectorSet sorted() {
        List<LabelSelector> sorted = selectors.stream()
            .sorted(Comparator.comparing(LabelSelector::getName))
            .collect(Collectors.toList());
        return new LabelSelectorSet(sorted);
    }

    public LinkedHashMap<String, String> toLinkedHashMap() {
        LinkedHashMap<String, String> r = new LinkedHashMap<>();
        forEachSelector(r::put);
        return r;
    }

    public Labels toLabels() {
        LabelsBuilder builder = Labels.builder();
        forEachSelector(builder::add);
        return builder.build();
    }

    public LabelSelector last() {
        return selectors.get(selectors.size() - 1);
    }

    public LabelSelectorSet onlyShardKey() {
        List<LabelSelector> filtered = selectors.stream()
                .filter(s -> LabelKeys.isShardKeyPart(s.getName()))
                .collect(Collectors.toList());
        return new LabelSelectorSet(filtered);
    }

    @Nullable
    public ShardKey getShardKey() {
        String project = null;
        String cluster = null;
        String service = null;
        for (LabelSelector selector : selectors) {
            String name = selector.getName();
            if (LabelKeys.PROJECT.equals(name)) {
                project = selector.getValueFormatted();
                continue;
            }

            if (LabelKeys.CLUSTER.equals(name)) {
                cluster = selector.getValueFormatted();
            }

            if (LabelKeys.SERVICE.equals(name)) {
                service = selector.getValueFormatted();
            }
        }

        if (project == null || cluster == null || service == null) {
            return null;
        }

        return new ShardKey(project, cluster, service);
    }

    public LabelSelectorSet withoutShardKey() {
        List<LabelSelector> filtered = selectors.stream()
                .filter(s -> !LabelKeys.isShardKeyPart(s.getName()))
                .collect(Collectors.toList());
        return new LabelSelectorSet(filtered);
    }

    public Labels filterEqConstSelectors() {
        LabelsBuilder builder = Labels.builder();
        for (LabelSelector selector : selectors) {
            Label label = selector.toParsed().asExactLabel();
            if (label != null) {
                builder.add(label);
            }
        }
        return builder.build();
    }

    public boolean isEmpty() {
        return selectors.isEmpty();
    }

    @Override
    public String toString() {
        return format();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        LabelSelectorSet that = (LabelSelectorSet) o;

        return selectors.equals(that.selectors);
    }

    @Override
    public int hashCode() {
        return selectors.hashCode();
    }
}
