package ru.yandex.solomon.labels.shard;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.BiConsumer;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.LabelKeys;


/**
 * @author Stepan Koltsov
 */
public class ShardKey implements Comparable<ShardKey> {
    @Nonnull
    private final String project;
    /** Cluster name (not cluster id) */
    @Nonnull
    private final String cluster;
    /** Service name (not service id) */
    @Nonnull
    private final String service;

    public ShardKey(@Nonnull String project, @Nonnull String cluster, @Nonnull String service) {
        if (project.isEmpty() || cluster.isEmpty() || service.isEmpty()) {
            throw new RuntimeException(
                    "wrong shard key: " +
                            "project=" + project + ", cluster=" + cluster + ", service=" + service);
        }
        this.project = project;
        this.cluster = cluster;
        this.service = service;
    }

    @Nonnull
    public String getProject() {
        return project;
    }

    @Nonnull
    public String getCluster() {
        return cluster;
    }

    @Nonnull
    public String getService() {
        return service;
    }

    public void forEachLabel(BiConsumer<String, String> keyValueConsumer) {
        keyValueConsumer.accept(LabelKeys.PROJECT, project);
        keyValueConsumer.accept(LabelKeys.CLUSTER, cluster);
        keyValueConsumer.accept(LabelKeys.SERVICE, service);
    }

    public Labels toLabels() {
        return Labels.of(LabelKeys.PROJECT, project, LabelKeys.CLUSTER, cluster, LabelKeys.SERVICE, service);
    }

    public ShardKeyAndMetricKey append(Labels key) {
        return new ShardKeyAndMetricKey(this, key);
    }

    public static ShardKey get(Labels labels) {
        String project = null;
        String cluster = null;
        String service = null;
        for (int index = 0; index < labels.size(); index++) {
            var label = labels.at(index);
            switch (label.getKey()) {
                case LabelKeys.PROJECT:
                    project = label.getValue();
                    break;
                case LabelKeys.CLUSTER:
                    cluster = label.getValue();
                    break;
                case LabelKeys.SERVICE:
                    service = label.getValue();
                    break;
            }
        }
        return create(project, cluster, service);
    }

    public static ShardKey get(Map<String, String> labels) {
        String project = labels.get(LabelKeys.PROJECT);
        String cluster = labels.get(LabelKeys.CLUSTER);
        String service = labels.get(LabelKeys.SERVICE);
        return create(project, cluster, service);
    }

    public static ShardKey create(@Nullable String project, @Nullable String cluster, @Nullable String service) {
        if (project == null || project.isEmpty()) {
            throw new RuntimeException("project is not specified");
        }
        if (cluster == null || cluster.isEmpty()) {
            throw new RuntimeException("cluster is not specified");
        }
        if (service == null || service.isEmpty()) {
            throw new RuntimeException("service is not specified");
        }
        return new ShardKey(project, cluster, service);
    }

    public static final String JOIN = ";";

    @Override
    public String toString() {
        return LabelKeys.PROJECT + '=' + project + JOIN +
            LabelKeys.CLUSTER + '=' + cluster + JOIN +
            LabelKeys.SERVICE + '=' + service;
    }

    public String toUniqueId() {
        return project + ':' + cluster + ':' + service;
    }

    @Override
    public int compareTo(@Nonnull ShardKey that) {
        int p = this.project.compareTo(that.project);
        if (p != 0) {
            return p;
        }
        int s = this.service.compareTo(that.service);
        if (s != 0) {
            return s;
        }
        int c = this.cluster.compareTo(that.cluster);
        if (c != 0) {
            return c;
        }
        return 0;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ShardKey shardKey = (ShardKey) o;

        if (!cluster.equals(shardKey.cluster)) return false;
        if (!project.equals(shardKey.project)) return false;
        if (!service.equals(shardKey.service)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = project.hashCode();
        result = 31 * result + cluster.hashCode();
        result = 31 * result + service.hashCode();
        return result;
    }

    public LinkedHashMap<String, String> toMap() {
        LinkedHashMap<String, String> map = new LinkedHashMap<>(3);
        map.put(LabelKeys.PROJECT, project);
        map.put(LabelKeys.CLUSTER, cluster);
        map.put(LabelKeys.SERVICE, service);
        return map;
    }
}
