package ru.yandex.solomon.main;

import java.io.PrintStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.base.Throwables;
import com.google.protobuf.Message;
import io.grpc.Server;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ConfigurableApplicationContext;
import sun.misc.Signal;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.config.SolomonConfigs;
import ru.yandex.solomon.config.protobuf.ELogToType;
import ru.yandex.solomon.config.protobuf.TLoggingConfig;
import ru.yandex.solomon.config.protobuf.http.HttpServerConfig;
import ru.yandex.solomon.main.http.HttpServer;
import ru.yandex.solomon.main.logger.DevNullPrintStream;
import ru.yandex.solomon.main.logger.LoggerConfigurationUtils;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.secrets.SecretProviders;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Sergey Polovko
 */
public abstract class SolomonWebApp {
    private static final Logger logger = LoggerFactory.getLogger(SolomonWebApp.class);

    private final Message configTemplate;
    private final AtomicBoolean terminating;
    private ConfigurableApplicationContext applicationCtx;
    private HttpServer httpServer;

    public SolomonWebApp(Message configTemplate) {
        this.configTemplate = configTemplate;
        this.terminating = new AtomicBoolean();
    }

    protected void run(String[] args) {
        PrintStream realStdout = System.out;
        PrintStream realStderr = System.err;

        Thread.setDefaultUncaughtExceptionHandler((t, e) -> {
            System.setOut(realStdout);
            System.setErr(realStderr);
            handleUncaughtException(t, e);
        });

        try {
            long startMillis = System.currentTimeMillis();
            Message config = parseConfig(args);

            ELogToType logTo = configureLogger(config);
            if (logTo != ELogToType.STDERR && logTo != ELogToType.STDOUT) {
                // redirect stdout and stderr to a /dev/null
                System.setOut(DevNullPrintStream.INSTANCE);
                System.setErr(DevNullPrintStream.INSTANCE);
            }

            var secretsProvider = parseSecrets(args);
            this.applicationCtx = SpringContexts.createWeb(getClass(), config, secretsProvider);

            HttpServerConfig httpServerConfig = applicationCtx.getBean("HttpServerConfig", HttpServerConfig.class);
            this.httpServer = new HttpServer(applicationCtx, httpServerConfig, "HttpServer");
            logger.info("{} started in {} ms", getClass().getSimpleName(), (System.currentTimeMillis() - startMillis));
            java.util.logging.Logger.getLogger(SolomonWebApp.class.getName()).info("java.util.logging binding is working");

            Signal.handle(new Signal("TERM"), sig -> {
                if (terminating.compareAndSet(false, true)) {
                    terminate();
                }
            });
        } catch (Throwable t) {
            System.setOut(realStdout);
            System.setErr(realStderr);

            t.printStackTrace();
            System.exit(1);
        }
    }

    protected Message parseConfig(String[] args) {
        String configFile = SolomonConfigs.getConfigFileFromArgs(args);
        return SolomonConfigs.parseConfig(configFile, configTemplate);
    }

    protected SecretProvider parseSecrets(String[] args) {
        Optional<String> secretsFile = SolomonConfigs.getSecretsFileFromArgs(args);
        if (secretsFile.isEmpty()) {
            logger.warn("parameter --secrets was not provided");
            return SecretProviders.inline();
        }

        var path = Path.of(secretsFile.get());
        if (!Files.exists(path)) {
            logger.warn("secret file {} doesn't exist", path);
            return SecretProviders.empty();
        }

        return SecretProviders.fromFile(path);
    }

    private static ELogToType configureLogger(Message config) {
        var loggingConfigField = config.getDescriptorForType().findFieldByName("LoggingConfig");
        TLoggingConfig loggingConfig = (TLoggingConfig) config.getField(loggingConfigField);
        LoggerConfigurationUtils.configureLogger(loggingConfig);
        return loggingConfig.getLogTo();
    }

    public static void handleUncaughtException(Thread t, Throwable x) {
        try {
            // Do not use `x.printStackTrace()` because otherwise stack trace
            // can be mixed with output from other threads.
            System.err.print(Instant.now().toString()
                    + " terminating on uncaught exception;"
                    + " thread=" + t.getName() + '\n'
                    + Throwables.getStackTraceAsString(x));
        } finally {
            // Halt instead of exit, because exit hooks may hang.
            //
            // 13 is chosen because it is unlikely used by anyone else,
            // so exit code can be used to quickly find out exit reason.

            Runtime.getRuntime().halt(13);
        }
    }

    private void terminate() {
        logger.info("Shutdown started");

        var httpServerClose = safeStop("http server", () -> httpServer.stop());
        var grpcServerClose = safeStop("grpc server", this::gracefulShutdownGrpcServers);
        var appClose = safeStop("app", this::gracefulShutdownApp);

        appClose.join();
        grpcServerClose.join();
        httpServerClose.join();
        applicationCtx.close();

        logger.info("Shutdown completed");
    }

    public CompletableFuture<Void> gracefulShutdownApp() {
        return applicationCtx.getBeansOfType(ApplicationShutdown.class)
                .entrySet()
                .stream()
                .map(entry -> safeStop(entry.getKey(), () -> entry.getValue().shutdown()))
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid));
    }

    public CompletableFuture<Void> gracefulShutdownGrpcServers() {
        return applicationCtx.getBeansOfType(Server.class)
                .entrySet()
                .stream()
                .map(entry -> {
                    return safeStop(entry.getKey(), () -> {
                        return CompletableFuture.runAsync(() -> {
                            var server = entry.getValue();
                            server.shutdown();
                            try {
                                if (!server.awaitTermination(3, TimeUnit.SECONDS)) {
                                    server.shutdownNow();
                                    server.awaitTermination(1, TimeUnit.SECONDS);
                                }
                            } catch (InterruptedException e) {
                                // ok
                            }
                        });
                    });
                })
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid));
    }

    public CompletableFuture<Void> safeStop(String what, Supplier<CompletableFuture<?>> supplier) {
        long startNs = System.nanoTime();
        logger.info("Shutdown started: {}", what);
        return CompletableFuture.completedFuture(null)
                .thenCompose(ignore -> supplier.get())
                .handle((o, e) -> {
                    var tookNs = System.nanoTime() - startNs;
                    logger.info("Shutdown completed: {}, took {}", what, DurationUtils.formatDurationNanos(tookNs), e);
                    return null;
                });
    }
}
