package ru.yandex.solomon.main;

import com.google.protobuf.Message;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.annotation.Order;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.context.ConfigurableWebApplicationContext;
import org.springframework.web.context.support.AnnotationConfigWebApplicationContext;
import org.springframework.web.reactive.config.DelegatingWebFluxConfiguration;

import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.secrets.SecretProviders;

/**
 * @author Sergey Polovko
 */
public class SpringContexts {

    public static ConfigurableApplicationContext createConfig(Message config, SecretProvider secrets) {
        var context = new GenericApplicationContext();
        var bf = context.getBeanFactory();
        bf.registerSingleton(config.getClass().getSimpleName(), config);
        config.getAllFields().forEach((field, value) -> {
            bf.registerSingleton(field.getName(), value);
        });
        bf.registerSingleton("secrets", secrets);
        context.refresh();
        return context;
    }

    public static ConfigurableApplicationContext createSimple(Class<?> annotatedConfig, Message config) {
        var context = new AnnotationConfigApplicationContext();
        context.setParent(createConfig(config, SecretProviders.empty()));
        context.register(annotatedConfig);
        context.refresh();
        return context;
    }

    public static ConfigurableWebApplicationContext createWeb(
            Class<?> annotatedConfig,
            Message config,
            SecretProvider secrets)
    {
        var context = new AnnotationConfigWebApplicationContext();
        context.setParent(createConfig(config, secrets));
        context.register(annotatedConfig);
        context.register(DelegatingWebFluxConfiguration.class);
        context.register(BinderControllerAdvice.class);
        context.refresh();
        return context;
    }

    @ControllerAdvice
    @Order(10000)
    public static class BinderControllerAdvice {
        @InitBinder
        public void setAllowedFields(WebDataBinder dataBinder) {
            // This code protects Spring Core from a "Remote Code Execution" attack (dubbed "Spring4Shell").
            // By applying this mitigation, you prevent the "Class Loader Manipulation" attack vector from firing.
            // For more details, see this post: https://www.lunasec.io/docs/blog/spring-rce-vulnerabilities/
            var denyList = new String[]{"class.*", "Class.*", "*.class.*", "*.Class.*"};
            dataBinder.setDisallowedFields(denyList);
        }
    }
}
