package ru.yandex.solomon.main.http;

import java.time.Duration;
import java.util.ArrayList;
import java.util.concurrent.CompletableFuture;

import io.netty.channel.Channel;
import io.netty.channel.EventLoopGroup;
import org.springframework.context.ApplicationContext;
import org.springframework.core.annotation.AnnotationAwareOrderComparator;
import org.springframework.http.codec.ServerCodecConfigurer;
import org.springframework.http.server.reactive.ReactorHttpHandlerAdapter;
import org.springframework.web.reactive.DispatcherHandler;
import org.springframework.web.server.WebExceptionHandler;
import org.springframework.web.server.WebFilter;
import org.springframework.web.server.adapter.WebHttpHandlerBuilder;
import reactor.netty.DisposableServer;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.http.HttpServerConfig;
import ru.yandex.solomon.selfmon.ng.JvmMon;
import ru.yandex.solomon.util.NettyUtils;

/**
 * @author Sergey Polovko
 */
public class HttpServer {

    private final DisposableServer server;
    private final EventLoopGroup selectorLoop;
    private final EventLoopGroup serverLoop;

    public HttpServer(ApplicationContext context, HttpServerConfig config, String serverName) {
        var httpHandler = WebHttpHandlerBuilder.webHandler(new DispatcherHandler(context))
                .exceptionHandlers(exceptionHandlers -> {
                    exceptionHandlers.addAll(context.getBeansOfType(WebExceptionHandler.class).values());
                })
                .filters(filters -> {
                    var additionalFilters = new ArrayList<>(context.getBeansOfType(WebFilter.class).values());
                    AnnotationAwareOrderComparator.sort(additionalFilters);
                    filters.addAll(additionalFilters);
                })
                .codecConfigurer(context.getBean(ServerCodecConfigurer.class))
                .build();

        {
            String selectorLoopName = serverName + "Selector";
            this.selectorLoop = NettyUtils.createEventLoopGroup(selectorLoopName, 1);
            JvmMon.addExecutorMetrics(selectorLoopName, selectorLoop, MetricRegistry.root());
        }

        this.serverLoop = NettyUtils.createEventLoopGroup(serverName, config.getThreadsCount());
        JvmMon.addExecutorMetrics(serverName, serverLoop, MetricRegistry.root());

        this.server = reactor.netty.http.server.HttpServer.create()
                .host(config.getBind())
                .port(config.getPort())
                .tcpConfiguration(tcpServer -> tcpServer.bootstrap(b -> {
                    b.group(selectorLoop, serverLoop);
                    b.channel(NettyUtils.serverChannelType());
                    return b;
                }))
                .handle(new ReactorHttpHandlerAdapter(httpHandler))
                .bindNow(Duration.ofSeconds(15));
    }

    public CompletableFuture<Void> stop() {
        Channel channel = server.channel();
        // stop accepting new connections first by closing server's channel
        return NettyUtils.toCompletableFuture(channel.close())
            // then shutdown underlying thread pools sequentially
            .thenCompose(aVoid -> NettyUtils.toCompletableFuture(selectorLoop.shutdownGracefully()))
            .thenCompose(aVoid -> NettyUtils.toCompletableFuture(serverLoop.shutdownGracefully()))
            .thenAccept(x -> {});
    }
}
