package ru.yandex.solomon.main.logger;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.AsyncAppender;
import org.apache.logging.log4j.core.appender.ConsoleAppender;
import org.apache.logging.log4j.core.appender.RollingFileAppender;
import org.apache.logging.log4j.core.config.AppenderRef;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.LoggerConfig;
import org.apache.logging.log4j.core.layout.PatternLayout;
import org.slf4j.bridge.SLF4JBridgeHandler;
import sun.misc.Signal;

import ru.yandex.monlib.metrics.log4j2.InstrumentedAppender;
import ru.yandex.solomon.config.protobuf.ELevel;
import ru.yandex.solomon.config.protobuf.ELogToType;
import ru.yandex.solomon.config.protobuf.TAppender;
import ru.yandex.solomon.config.protobuf.TLogger;
import ru.yandex.solomon.config.protobuf.TLoggingConfig;


/**
 * @author alexlovkov
 * @author Sergey Polovko
 */
public class LoggerConfigurationUtils {

    public static final String ROLLING_SIGNAL = "USR2";

    public static void configureLogger(TLoggingConfig config) {
        SLF4JBridgeHandler.removeHandlersForRootLogger();
        SLF4JBridgeHandler.install();
        if ("".equals(config.getLogFile()) && config.getLoggersCount() == 0) {
            throw new IllegalArgumentException("config doesn't contain logger configuration");
        }

        Configurator c = new Configurator();
        c.clear();
        c.setDefaultPatternLayout("%d %-5p %t/%c{1.}: %m%n");

        var defaultOut = c.addDefaultAppender(config.getLogTo(), config.getLogFile());
        Map<String, AsyncAppender> appenders = new HashMap<>(config.getAppendersCount());
        for (var conf : config.getAppendersList()) {
            var appender = c.addAppender(conf);
            appenders.put(conf.getName(), appender);
        }

        for (TLogger loggerConfig : config.getLoggersList()) {
            Level level = Level.getLevel(loggerConfig.getLevel().name());
            if (level == null) {
                throw new IllegalStateException("unknown logger level in config: " + loggerConfig.getLevel());
            }

            var appenderName = loggerConfig.getAppender();
            var appender = appenders.getOrDefault(appenderName, defaultOut);

            if ("root".equals(loggerConfig.getName())) {
                c.configureRootLogger(level, appender);
            } else {
                c.addLogger(loggerConfig.getName(), level, appender);
            }
        }

        c.finish();
    }

    public static void simpleLogger(Level level) {
        configureLogger(TLoggingConfig.newBuilder()
            .addLoggers(TLogger.newBuilder().setName("root").setLevel(ELevel.valueOf(level.name())).build())
            .build());
    }

    public static void disableLogger() {
        Configurator c = new Configurator();
        c.clear();
        c.finish();
    }

    /**
     * Log4j2 configurator helper.
     */
    private static final class Configurator {
        private final LoggerContext context;
        private final Configuration log4jConfig;
        private final List<SignalTriggeringPolicy> rollupPolicies = new CopyOnWriteArrayList<>();
        private PatternLayout defaultLayout = PatternLayout.createDefaultLayout();

        Configurator() {
            this.context = LoggerContext.getContext(false);
            this.log4jConfig = context.getConfiguration();
            // jvm crashes if you send USR2 signal and don't have handler for it
            Signal.handle(new Signal(ROLLING_SIGNAL), signal -> rollupPolicies.forEach(SignalTriggeringPolicy::onSignal));
        }

        void clear() {
            for (String name : log4jConfig.getLoggers().keySet()) {
                log4jConfig.removeLogger(name);
            }
            LoggerConfig rootLogger = log4jConfig.getRootLogger();
            rootLogger.getAppenders().keySet().forEach(rootLogger::removeAppender);
        }

        void setDefaultPatternLayout(String pattern) {
            this.defaultLayout = createPatternLayout(pattern);
        }

        PatternLayout createPatternLayout(String pattern) {
            return PatternLayout.newBuilder()
                .withPattern(pattern)
                .withConfiguration(log4jConfig)
                .build();
        }

        AppenderRef addOutAppender(String name, ELogToType logTo, String logFile, PatternLayout layout) {
            AppenderRef ref = AppenderRef.createAppenderRef(name, null, null);
            if (logTo == ELogToType.FILE) {
                File f = new File(logFile);
                if (f.exists() && !f.canWrite()) {
                    throw new IllegalStateException("can't write to the log file:" + logFile);
                }
                try {
                    boolean ignore = f.createNewFile();
                } catch (IOException e) {
                    throw new IllegalStateException(
                        "log file:" + logFile + " doesn't exist, and we don't have permission to create it", e);
                }

                SignalTriggeringPolicy rollupPolicy = new SignalTriggeringPolicy();
                rollupPolicies.add(rollupPolicy);
                log4jConfig.addAppender(RollingFileAppender.newBuilder()
                    .setName(ref.getRef())
                    .withFileName(logFile)
                    .withAppend(true)
                    .setIgnoreExceptions(false)
                    .withStrategy(new SignalRolloverStrategy())
                    .withPolicy(rollupPolicy)
                    .withFilePattern("")
                    .withImmediateFlush(false)
                    .withBufferedIo(true)
                    .withBufferSize(64 << 10) // 64 KiB
                    .setLayout(layout)
                    .setConfiguration(log4jConfig)
                    .build());
            } else {

                if (logTo == ELogToType.STDERR) {
                    log4jConfig.addAppender(ConsoleAppender.newBuilder()
                        .setName(ref.getRef())
                        .setTarget(ConsoleAppender.Target.SYSTEM_ERR)
                        .setFollow(true)
                        .setLayout(layout)
                        .setConfiguration(log4jConfig)
                        .build());

                } else if (logTo == ELogToType.STDOUT) {
                    log4jConfig.addAppender(ConsoleAppender.newBuilder()
                        .setName(ref.getRef())
                        .setTarget(ConsoleAppender.Target.SYSTEM_OUT)
                        .setFollow(true)
                        .setLayout(layout)
                        .setConfiguration(log4jConfig)
                        .build());

                } else {
                    throw new IllegalArgumentException("unknown log destination type: " + logTo);
                }
            }
            return ref;
        }

        AppenderRef addMetricsAppender(String name, PatternLayout layout) {
            AppenderRef ref = AppenderRef.createAppenderRef(name + "_sensors", null, null);
            log4jConfig.addAppender(InstrumentedAppender.createAppender(ref.getRef(), true, layout, null));
            return ref;
        }

        AsyncAppender addAsyncAppender(AppenderRef... refs) {
            AsyncAppender asyncAppender = AsyncAppender.newBuilder()
                .setName(refs[0].getRef() + "_async")
                .setBlocking(false)
                .setBufferSize(64 << 10) // 64 KiB
                .setAppenderRefs(refs)
                .setConfiguration(log4jConfig)
                .build();
            log4jConfig.addAppender(asyncAppender);
            return asyncAppender;
        }

        AsyncAppender addDefaultAppender(ELogToType logTo, String logFile) {
            return addAppender(TAppender.newBuilder()
                .setName("out")
                .setLogTo(logTo)
                .setLogFile(logFile)
                .build());
        }

        AsyncAppender addAppender(TAppender config) {
            var layout = config.getPatternLayout().isEmpty()
                ? defaultLayout
                : createPatternLayout(config.getPatternLayout());

            var metrics = addMetricsAppender(config.getName(), layout);
            var out = addOutAppender(config.getName(), config.getLogTo(), config.getLogFile(), layout);
            return addAsyncAppender(out, metrics);
        }

        void configureRootLogger(Level level, Appender asyncAppender) {
            LoggerConfig rootLogger = log4jConfig.getRootLogger();
            rootLogger.setAdditive(false);
            rootLogger.setLevel(level);
            rootLogger.addAppender(asyncAppender, null, null);
        }

        void addLogger(String name, Level level, AsyncAppender appender) {
            var ref = AppenderRef.createAppenderRef(appender.getName(), null, null);
            LoggerConfig logger = LoggerConfig.createLogger(
                false, level, name, "false", new AppenderRef[]{ref}, null, log4jConfig, null);
            logger.addAppender(appender, null, null);
            log4jConfig.addLogger(logger.getName(), logger);
        }

        void finish() {
            log4jConfig.start();
            context.updateLoggers();
        }
    }
}
