package ru.yandex.solomon.memory.layout;


import io.netty.buffer.ByteBuf;
import io.netty.buffer.UnpooledByteBufAllocator;
import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.junit.Assert;
import org.junit.Test;
import org.openjdk.jol.info.GraphLayout;
import org.openjdk.jol.vm.VM;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class MemoryCounterTest {

    @Test
    public void arrayByteSize() {
        byte[] source = new byte[1042];
        assertEquals(VM.current().sizeOf(source), MemoryCounter.arrayObjectSize(source));
    }

    @Test
    public void arrayIntSize() {
        int[] source = new int[123];
        assertEquals(VM.current().sizeOf(source), MemoryCounter.arrayObjectSize(source));
    }

    @Test
    public void arrayLongSize() {
        long[] source = new long[4455];
        assertEquals(VM.current().sizeOf(source), MemoryCounter.arrayObjectSize(source));
    }

    @Test
    public void arrayDoubleSize() {
        double[] source = new double[4455];
        assertEquals(VM.current().sizeOf(source), MemoryCounter.arrayObjectSize(source));
    }

    @Test
    public void arrayObjectSize() {
        Object[] source = new Object[123];
        assertEquals(VM.current().sizeOf(source), MemoryCounter.arrayObjectSize(source));
    }

    @Test
    public void log2ObjectOpenHashMapEmptySize() {
        Long2ObjectOpenHashMap source = new Long2ObjectOpenHashMap();
        GraphLayout gl = GraphLayout.parseInstance(source);

        assertEquals(gl.toPrintable(), gl.totalSize(), MemoryCounter.long2ObjectOpenHashMapSize(source));
    }

    @Test
    public void int2LongOpenHashMapEmptySize() {
        var source = new Int2LongOpenHashMap();
        GraphLayout gl = GraphLayout.parseInstance(source);

        assertEquals(gl.toPrintable(), gl.totalSize(), MemoryCounter.int2LongMapSize(source));
    }

    @Test
    public void long2ObjectOpenHashMapSize() {
        Long2ObjectOpenHashMap<Object> source = new Long2ObjectOpenHashMap<>();
        for (int index = 0; index < 123; index++) {
            source.put(index, null);
        }

        GraphLayout gl = GraphLayout.parseInstance(source);
        assertEquals(gl.totalSize(), MemoryCounter.long2ObjectOpenHashMapSize(source));
    }

    @Test
    public void int2LongOpenHashMapSize() {
        var source = new Int2LongOpenHashMap();
        for (int index = 0; index < 123; index++) {
            source.put(index, index);
        }

        GraphLayout gl = GraphLayout.parseInstance(source);

        assertEquals(gl.toPrintable(), gl.totalSize(), MemoryCounter.int2LongMapSize(source));
    }

    @Test
    public void stringSize() {
        for (long l = 1; l > 0; l *= 10) {
            String s = Long.toString(l);
            long estimated = MemoryCounter.stringSize(s);
            long precise = GraphLayout.parseInstance(s).totalSize();
            Assert.assertEquals(s, precise, estimated);
        }
    }

    @Test
    public void byteBufSize() {
        ByteBuf buffer = UnpooledByteBufAllocator.DEFAULT.heapBuffer(10 << 20);
        long size = MemoryCounter.byteBufSize(buffer);
        long estimated = GraphLayout.parseInstance(buffer).totalSize() - GraphLayout.parseInstance(UnpooledByteBufAllocator.DEFAULT).totalCount();
        assertEquals(estimated, size, 400);
    }
}
