package ru.yandex.metabase.client;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.google.common.collect.Iterables;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.discovery.FilterDiscoveryService;
import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.metabase.client.TMetabaseClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;

/**
 * @author Vladimir Gordiychuk
 */
public class MetabaseClientFactory {
    private final ThreadPoolProvider threadPoolProvider;
    private final ClusterMapper clusterMapper;
    private final MetricRegistry registry;
    private final ClientOptionsFactory clientOptionsFactory;

    public MetabaseClientFactory(
            ThreadPoolProvider threadPoolProvider,
            ClusterMapper clusterMapper,
            MetricRegistry registry,
            ClientOptionsFactory clientOptionsFactory)
    {
        this.threadPoolProvider = threadPoolProvider;
        this.clusterMapper = clusterMapper;
        this.registry = registry;
        this.clientOptionsFactory = clientOptionsFactory;
    }

    public MetabaseClient createClient(String clientId, TMetabaseClientConfig config) {
        return Iterables.getOnlyElement(createClients(clientId, config).entrySet()).getValue();
    }

    public Map<String, MetabaseClient> createClients(String clientId, TMetabaseClientConfig config) {
        List<String> addresses = config.getGrpcConfig().getAddressesList();
        Map<String, MetabaseClient> result = new HashMap<>();
        for (String clusterId : clusterMapper.knownClusterIds()) {
            var discovery = new FilterDiscoveryService(DiscoveryService.async(), (address) -> {
                return Objects.equals(clusterId, clusterMapper.byFqdnOrNull(address.getHost()));
            });

            var options = MetabaseClientOptions.newBuilder(
                            clientOptionsFactory.newBuilder(
                            "MetabaseClientConfig",
                            config.getGrpcConfig())
                        .setClientId(clientId)
                        .setMetricRegistry(registry.subRegistry("target", clusterId)))
                .setFromConfig(config)
                .setDiscoveryService(discovery)
                .build();

            MetabaseClient client = MetabaseClients.createDynamic(addresses, options);
            result.put(clusterId, client);
        }
        return result;
    }
}
