package ru.yandex.metabase.client;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.grpc.utils.GrpcClientOptions;
import ru.yandex.solomon.config.protobuf.metabase.client.TMetabaseClientConfig;

import static ru.yandex.solomon.config.OptionalSet.setTime;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class MetabaseClientOptions {
    private static final long DEFAULT_PREFETCH_EXPIRE_MILLIS = TimeUnit.MINUTES.toMillis(10L);
    private static final long DEFAULT_PREFETCH_RETRY_DELAY_MILLIS = TimeUnit.SECONDS.toMillis(3L);

    private static final MetabaseClientOptions EMPTY = newBuilder(DefaultClientOptions.empty()).build();

    private final GrpcClientOptions grpcOptions;

    private final long metadataExpireMillis;
    private final long metadataRetryDelayMillis;
    private final long metadataRequestTimeOut;
    private final DiscoveryService discoveryService;

    private MetabaseClientOptions(Builder builder) {
        this.grpcOptions = builder.grpcOptions;

        this.metadataExpireMillis = builder.metadataExpireMillis;
        this.metadataRetryDelayMillis = builder.metadataRetryDelayMillis;
        this.metadataRequestTimeOut = builder.metadataRequestTimeoutMs == null
                ? grpcOptions.getDefaultTimeoutMillis()
                : builder.metadataRequestTimeoutMs;
        this.discoveryService = Optional.ofNullable(builder.discoveryService).orElseGet(DiscoveryService::async);
    }

    public static Builder newBuilder(GrpcClientOptions.Builder grpcOptionsBuilder) {
        return newBuilder(grpcOptionsBuilder.build());
    }

    public static Builder newBuilder(GrpcClientOptions grpcOptions) {
        return new Builder(grpcOptions);
    }

    public static MetabaseClientOptions empty() {
        return EMPTY;
    }

    public GrpcClientOptions getGrpcOptions() {
        return grpcOptions;
    }

    public long getMetadataExpireMillis() {
        return metadataExpireMillis;
    }

    public long getMetadataRetryDelayMillis() {
        return metadataRetryDelayMillis;
    }

    public long getMetadataRequestTimeOut() {
        return metadataRequestTimeOut;
    }

    public DiscoveryService getDiscoveryService() {
        return discoveryService;
    }

    @ParametersAreNonnullByDefault
    public static final class Builder {
        private final GrpcClientOptions grpcOptions;

        private long metadataExpireMillis = DEFAULT_PREFETCH_EXPIRE_MILLIS;
        private long metadataRetryDelayMillis = DEFAULT_PREFETCH_RETRY_DELAY_MILLIS;
        @Nullable
        private Long metadataRequestTimeoutMs;
        @Nullable
        private DiscoveryService discoveryService;

        public Builder(GrpcClientOptions grpcOptions) {
            this.grpcOptions = grpcOptions;
        }

        /**
         * Define how much time metadata from cluster will be relevant.
         * Client will be automatically refresh metadata with specified delay.
         * <p>
         * Too high delay can lead to obsolete labelList and as a result search will
         * not find new metrics, but too low delay lead to many requests to server.
         */
        public Builder setExpireClusterMetadata(long value, TimeUnit unit) {
            if (value <= 0) {
                throw new IllegalArgumentException("Expire time can not be negative: " + value);
            }

            this.metadataExpireMillis = unit.toMillis(value);
            return this;
        }

        /**
         * Background update local label index can fail on particular node. This parameter define retry delay that
         * increments exponential for sequential fails on particular node. If parameter not specified
         * will be use default delay equal to 3s.
         */
        public Builder setClusterMetadataRequestRetryDelay(long time, TimeUnit unit) {
            this.metadataRetryDelayMillis = unit.toMillis(time);
            return this;
        }

        /**
         * Timeout for metadata requests. This parameters also define max latency between background
         * metadata refresh. It means that next metadata refresh occurs only when collected answer
         * from each node, and if one of the node failed and not answer background process will wait
         * time out.
         * <p>
         * It's an optional parameter, If the user has not provided timeout when the client is build,
         * client will use default request timeout ({@link GrpcClientOptions#getDefaultTimeoutMillis()})
         */
        public Builder setMetaDataRequestTimeOut(long value, TimeUnit unit) {
            this.metadataRequestTimeoutMs = unit.toMillis(value);
            return this;
        }

        public Builder setDiscoveryService(@Nullable DiscoveryService discoveryService) {
            this.discoveryService = discoveryService;
            return this;
        }

        public Builder setFromConfig(TMetabaseClientConfig config) {
            setTime(this::setExpireClusterMetadata, config.getMetadataExpireTime());
            setTime(this::setMetaDataRequestTimeOut, config.getMetadataRequestTimeout());
            setTime(this::setClusterMetadataRequestRetryDelay, config.getMetadataRetryDelayTimeout());
            return this;
        }

        public MetabaseClientOptions build() {
            return new MetabaseClientOptions(this);
        }
    }
}
