package ru.yandex.metabase.client.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.function.BiFunction;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillCloseWhenClosed;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.thread.executor.SyncExecutor;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindResponse;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsResponse;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsResponse;
import ru.yandex.solomon.model.protobuf.Selector;
import ru.yandex.solomon.selfmon.AvailabilityStatus;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class GrpcMetabaseClient implements MetabaseClient, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(GrpcMetabaseClient.class);

    @WillCloseWhenClosed
    private final MetabaseCluster cluster;
    private final ExecutorService responseHandlerExecutorService;
    private final boolean ownResponseHandlerExecutorService;

    public GrpcMetabaseClient(List<String> addresses, MetabaseClientOptions options) {
        this.cluster = new MetabaseCluster(addresses, options);

        Optional<ExecutorService> responseHandler = options.getGrpcOptions().getResponseHandlerExecutorService();
        if (responseHandler.isPresent()) {
            this.responseHandlerExecutorService = responseHandler.get();
            this.ownResponseHandlerExecutorService = false;
        } else {
            this.ownResponseHandlerExecutorService = true;
            this.responseHandlerExecutorService = new SyncExecutor();
        }
    }

    @Override
    public CompletableFuture<CreateOneResponse> createOne(CreateOneRequest request) {
        return internalCreateOne(request).thenCompose(response -> {
            if (response.getStatus() == EMetabaseStatusCode.SHARD_NOT_FOUND) {
                return cluster.forceUpdateClusterState()
                    .thenCompose(forced -> internalCreateOne(request));
            }
            return CompletableFuture.completedFuture(response);
        });
    }

    private CompletableFuture<CreateOneResponse> internalCreateOne(CreateOneRequest request) {
        return sendSeparateRequestToPartitionedShards(request, CreateOneResponse.class,
                input -> PartitionedShardResolver.of(cluster).resolvePartitionByLabels(request.getMetric().getLabelsList(), true)
                        .collect(Collectors.toMap(
                                Function.identity(),
                                partitionKey -> MetabaseRequests.withPartitionKey(CreateOneRequest.class, request, partitionKey.shard(), partitionKey.partitionId())
                        )),
                (node, input) -> node.createOne(input),
                MetabaseResponses::reduce,
                () -> MetabaseResponses.shardNotFoundForOne(CreateOneResponse.class)
                        .apply("Shard not found for metric: " + request.getMetric())
        );
    }

    @Override
    public CompletableFuture<CreateManyResponse> createMany(CreateManyRequest request) {
        return internalCreateMany(request).thenCompose(response -> {
            if (response.getStatus() == EMetabaseStatusCode.SHARD_NOT_FOUND) {
                return cluster.forceUpdateClusterState()
                    .thenCompose(forced -> internalCreateMany(request));
            }
            return CompletableFuture.completedFuture(response);
        });
    }

    private CompletableFuture<CreateManyResponse> internalCreateMany(CreateManyRequest request) {
        if (request.getMetricsCount() == 0) {
            return MetabaseResponses.completedException(CreateManyResponse.class,
                EMetabaseStatusCode.INVALID_REQUEST,
                "Request not contain metrics"
            );
        }

        return sendRequestToPartitionShards(request, CreateManyRequest.class, CreateManyResponse.class,
                () -> PartitionedShardResolver.of(cluster).resolveByLabels(request.getCommonLabelsList()),
                (node, input) -> node.createMany(input),
                MetabaseResponses::reduce,
                MetabaseResponses.shardNotFoundForMany(CreateManyResponse.class)
        );
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        return internalResolveOne(request).thenCompose(response -> {
            if (response.getStatus() == EMetabaseStatusCode.SHARD_NOT_FOUND) {
                return cluster.forceUpdateClusterState()
                    .thenCompose(forced -> internalResolveOne(request));
            }
            return CompletableFuture.completedFuture(response);
        });
    }

    private CompletableFuture<ResolveOneResponse> internalResolveOne(ResolveOneRequest request) {
        return sendSeparateRequestToPartitionedShards(request, ResolveOneResponse.class,
                input -> PartitionedShardResolver.of(cluster).resolvePartitionByLabels(request.getLabelsList(), true)
                        .collect(Collectors.toMap(
                                Function.identity(),
                                partitionKey -> MetabaseRequests.withPartitionKey(ResolveOneRequest.class, request, partitionKey.shard(), partitionKey.partitionId())
                        )),
                (node, input) -> node.resolveOne(input),
                MetabaseResponses::reduce,
                () -> MetabaseResponses.shardNotFoundForOne(ResolveOneResponse.class)
                        .apply("Shard not found for metric: " + request.getLabelsList())
        );
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        return internalResolveMany(request).thenCompose(response -> {
            if (response.getStatus() == EMetabaseStatusCode.SHARD_NOT_FOUND) {
                return cluster.forceUpdateClusterState()
                    .thenCompose(forced -> internalResolveMany(request));
            }
            return CompletableFuture.completedFuture(response);
        });
    }

    private CompletableFuture<ResolveManyResponse> internalResolveMany(ResolveManyRequest request) {
        if (request.getListLabelsCount() == 0) {
            return MetabaseResponses.completedException(ResolveManyResponse.class,
                    EMetabaseStatusCode.INVALID_REQUEST,
                    "Request not contain metrics");
        }
        return sendRequestToPartitionShards(request, ResolveManyRequest.class, ResolveManyResponse.class,
                () -> PartitionedShardResolver.of(cluster).resolveByLabels(request.getCommonLabelsList()),
                (node, input) -> node
                        .resolveMany(input)
                        .thenApplyAsync(Function.identity(), responseHandlerExecutorService),
                MetabaseResponses::reduce,
                MetabaseResponses.shardNotFoundForMany(ResolveManyResponse.class)
        );
    }

    @Override
    public CompletableFuture<DeleteManyResponse> deleteMany(DeleteManyRequest request) {
        return internalDeleteMany(request).thenCompose(response -> {
            if (response.getStatus() == EMetabaseStatusCode.SHARD_NOT_FOUND) {
                return cluster.forceUpdateClusterState()
                    .thenCompose(forced -> internalDeleteMany(request));
            }
            return CompletableFuture.completedFuture(response);
        });
    }

    private CompletableFuture<DeleteManyResponse> internalDeleteMany(DeleteManyRequest request) {
        return sendSeparateRequestToPartitionedShards(request, DeleteManyResponse.class,
                input -> {
                    var shardResolver = PartitionedShardResolver.of(cluster);
                    Map<PartitionKey, DeleteManyRequest.Builder> partitionRequests = new HashMap<>(1);
                    for (Metric metric : input.getMetricsList()) {
                        shardResolver.resolvePartitionByLabels(metric.getLabelsList(), true)
                                .forEach(partitionKey -> {
                                    var builder = partitionRequests.computeIfAbsent(partitionKey,
                                            k -> MetabaseRequests.builderWithPartitionKey(
                                                    DeleteManyRequest.class,
                                                    DeleteManyRequest.newBuilder().setDeadlineMillis(request.getDeadlineMillis()),
                                                    partitionKey.shard(),
                                                    partitionKey.partitionId()
                                            ));
                                    builder.addMetrics(metric);
                                });
                    }
                    return partitionRequests.entrySet().stream().collect(
                                toMap(e -> e.getKey(), entry -> entry.getValue().build())
                            );
                },
                (node, input) -> node.deleteMany(input),
                MetabaseResponses::reduce,
                MetabaseResponses.shardNotFoundForMany(DeleteManyResponse.class)
        );
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        return internalFind(request).thenCompose(response -> {
            if (response.getStatus() == EMetabaseStatusCode.SHARD_NOT_FOUND) {
                return cluster.forceUpdateClusterState()
                        .thenCompose(forced -> internalFind(request));
            }
            return CompletableFuture.completedFuture(response);
        });
    }

    private CompletableFuture<FindResponse> internalFind(FindRequest request) {
        final String findRequestForMultipleShardsError = "Slice offset doesn't implemented for cross-nodes requests";
        if (hasOffsetRestrictions(request)) {
            return MetabaseResponses.completedException(FindResponse.class,
                    EMetabaseStatusCode.INVALID_REQUEST,
                    findRequestForMultipleShardsError);
        }

        return sendRequestToPartitionShards(request, FindRequest.class, FindResponse.class,
                () -> {
                    final Selectors selectors = request.hasNewSelectors()
                            ? LabelSelectorConverter.protoToSelectors(request.getNewSelectors())
                            : LabelSelectorConverter.protoToSelectors(request.getSelectorsList());
                    var shardResolver = PartitionedShardResolver.of(cluster);
                    var shardStream = shardResolver.resolve(selectors);
                    if (hasSliceOptions(request)) {
                        var nodes = shardStream.collect(toList());
                        if (nodes.size() > 1) {
                            throw MetabaseResponses.createException(
                                    findRequestForMultipleShardsError,
                                    EMetabaseStatusCode.INVALID_REQUEST
                            );
                        }
                        return nodes.stream();
                    }
                    return shardStream;
                },
                (node, input) -> node.find(input),
                MetabaseResponses.reduce(request),
                MetabaseResponses.emptyOkResponse(FindResponse.class)
        );
    }

    private static boolean hasSliceOptions(FindRequest request) {
        return request.hasSliceOptions() && request.getSliceOptions().getOffset() > 0;
    }

    private static boolean hasOffsetRestrictions(FindRequest request) {
        if (hasSliceOptions(request)) {
            final List<Selector> protoSelectors;

            if (request.getNewSelectors().getLabelSelectorsCount() > 0) {
                protoSelectors = request.getNewSelectors().getLabelSelectorsList();
            } else {
                protoSelectors = request.getSelectorsList();
            }
            return !isSingleShardRequest(protoSelectors);
        }

        return false;
    }

    private static boolean isSingleShardRequest(List<Selector> protoSelectors) {
        return ShardSelectors.isSingleShard(LabelSelectorConverter.protoToSelectors(protoSelectors));
    }

    @Override
    public CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request) {
        return internalMetricNames(request).thenCompose(response -> {
            if (response.getStatus() == EMetabaseStatusCode.SHARD_NOT_FOUND) {
                return cluster.forceUpdateClusterState()
                    .thenCompose(forced -> internalMetricNames(request));
            }
            return CompletableFuture.completedFuture(response);
        });
    }

    private CompletableFuture<MetricNamesResponse> internalMetricNames(MetricNamesRequest request) {
        return sendRequestToPartitionShards(request, MetricNamesRequest.class, MetricNamesResponse.class,
                () -> {
                    Selectors selectors = LabelSelectorConverter.protoToSelectors(request.getSelectors());
                    return PartitionedShardResolver.of(cluster).resolve(selectors);
                },
                (node, input) -> node.metricNames(input),
                MetabaseResponses.reduce(request),
                MetabaseResponses.shardNotFoundForMany(MetricNamesResponse.class)
        );
    }

    @Override
    public CompletableFuture<TLabelValuesResponse> labelValues(TLabelValuesRequest request) {
        return internalLabelValues(request).thenCompose(response -> {
            if (response.getStatus() == EMetabaseStatusCode.SHARD_NOT_FOUND) {
                return cluster.forceUpdateClusterState()
                    .thenCompose(forced -> internalLabelValues(request));
            }
            return CompletableFuture.completedFuture(response);
        });
    }

    private CompletableFuture<TLabelValuesResponse> internalLabelValues(TLabelValuesRequest request) {
        return sendRequestToPartitionShards(request, TLabelValuesRequest.class, TLabelValuesResponse.class,
                () -> {
                    Selectors selectors = request.hasNewSelectors()
                            ? LabelSelectorConverter.protoToSelectors(request.getNewSelectors())
                            : LabelSelectorConverter.protoToSelectors(request.getSelectorsList());
                    return PartitionedShardResolver.of(cluster).resolve(selectors);
                },
                (node, input) -> node.labelValues(input),
                MetabaseResponses.reduce(request),
                MetabaseResponses.shardNotFoundForMany(TLabelValuesResponse.class)
        );
    }

    @Override
    public CompletableFuture<TLabelNamesResponse> labelNames(TLabelNamesRequest request) {
        return internalLabelNames(request).thenCompose(response -> {
            if (response.getStatus() == EMetabaseStatusCode.SHARD_NOT_FOUND) {
                return cluster.forceUpdateClusterState()
                    .thenCompose(forced -> internalLabelNames(request));
            }
            return CompletableFuture.completedFuture(response);
        });
    }

    private CompletableFuture<TLabelNamesResponse> internalLabelNames(TLabelNamesRequest request) {
        return sendRequestToPartitionShards(request, TLabelNamesRequest.class, TLabelNamesResponse.class,
                () -> {
                    Selectors selectors = request.hasNewSelectors()
                            ? LabelSelectorConverter.protoToSelectors(request.getNewSelectors())
                            : LabelSelectorConverter.protoToSelectors(request.getSelectorsList());
                    return PartitionedShardResolver.of(cluster).resolve(selectors);
                },
                (node, input) -> node.labelNames(input),
                MetabaseResponses::reduce,
                MetabaseResponses.shardNotFoundForMany(TLabelNamesResponse.class)
        );
    }

    @Override
    public CompletableFuture<TUniqueLabelsResponse> uniqueLabels(TUniqueLabelsRequest request) {
        return internalUniqueLabels(request).thenCompose(response -> {
            if (response.getStatus() == EMetabaseStatusCode.SHARD_NOT_FOUND) {
                return cluster.forceUpdateClusterState()
                    .thenCompose(forced -> internalUniqueLabels(request));
            }
            return CompletableFuture.completedFuture(response);
        });
    }

    private CompletableFuture<TUniqueLabelsResponse> internalUniqueLabels(TUniqueLabelsRequest request) {
        return sendRequestToPartitionShards(request, TUniqueLabelsRequest.class, TUniqueLabelsResponse.class,
                () -> {
                    Selectors selectors = request.hasNewSelectors()
                            ? LabelSelectorConverter.protoToSelectors(request.getNewSelectors())
                            : LabelSelectorConverter.protoToSelectors(request.getSelectorsList());
                    return PartitionedShardResolver.of(cluster).resolve(selectors);
                },
                (node, input) -> node.uniqueLabels(input),
                MetabaseResponses::reduce,
                MetabaseResponses.shardNotFoundForMany(TUniqueLabelsResponse.class)
        );
    }

    @Override
    public CompletableFuture<TResolveLogsResponse> resolveLogs(TResolveLogsRequest request) {
        return sendRequestToPartitionShards(request, TResolveLogsRequest.class, TResolveLogsResponse.class,
                () -> PartitionedShardResolver.of(cluster).resolveByNumId(request.getNumId(), true),
                (node, input) -> node.resolveLogs(input),
                MetabaseResponses::reduce,
                () -> MetabaseResponses.shardNotFoundForOne(TResolveLogsResponse.class)
                        .apply("Shard " + Integer.toUnsignedLong(request.getNumId()) + " location is unknown")
        );
    }

    @Override
    public CompletableFuture<Void> forceUpdateClusterMetaData() {
        return cluster.forceUpdateClusterState();
    }

    @Override
    public AvailabilityStatus getAvailability() {
        return cluster.getAvailability();
    }

    @Override
    public boolean isAllowCreateNew(int numId) {
        return cluster.isAllowCreateNew(numId);
    }

    @Override
    public Stream<Labels> shards(Selectors selectors) {
        return PartitionedShardResolver.of(cluster)
                .allShards(selectors)
                .map(PartitionedShard::getKey)
                .map(ShardKey::toLabels);
    }

    @Override
    public void close() {
        this.cluster.close();
        if (ownResponseHandlerExecutorService) {
            responseHandlerExecutorService.shutdown();
        }
    }

    private <Input, Output> CompletableFuture<Output> sendRequestToPartitionShards(
            Input request,
            Class<Input> inputClass,
            Class<Output> outputClass,
            Supplier<Stream<? extends PartitionedShard>> nodesGet,
            BiFunction<MetabaseNodeClient, Input, CompletableFuture<Output>> nodeMethod,
            BinaryOperator<Output> reducer,
            Supplier<Output> empty) {
        try {
            var futureStream = nodesGet.get().flatMap(partitionedShard ->
                partitionedShard.partitionStream().mapToObj(partitionId -> {
                            var fqdn = partitionedShard.getFqdn(partitionId);
                            var nodeClient = cluster.getNode(fqdn);
                            var partitionRequest = MetabaseRequests.withPartitionKey(inputClass, request, partitionedShard, partitionId);
                            return nodeMethod.apply(nodeClient, partitionRequest);
                        })
            );
            return accumulateOutputs(outputClass, futureStream, reducer, empty);
        } catch (Throwable throwable) {
            return CompletableFuture.completedFuture(
                    MetabaseResponses.exceptionResponse(outputClass, throwable, logger)
            );
        }
    }

    private <RequestT, ResponseT> CompletableFuture<ResponseT> sendSeparateRequestToPartitionedShards(
            RequestT request,
            Class<ResponseT> outputClass,
            Function<RequestT, Map<PartitionKey, RequestT>> splitRequestToPartitionParts,
            BiFunction<MetabaseNodeClient, RequestT, CompletableFuture<ResponseT>> nodeMethod,
            BinaryOperator<ResponseT> reducer,
            Supplier<ResponseT> empty) {
        try {
            var futureStream = splitRequestToPartitionParts.apply(request).entrySet().stream()
                    .map(entry -> {
                        var partitionKey = entry.getKey();
                        var requestPart = entry.getValue();
                        var fqdn = partitionKey.shard().getFqdn(partitionKey.partitionId());
                        var nodeClient = cluster.getNode(fqdn);
                        return nodeMethod.apply(nodeClient, requestPart);
                    }
            );
            return accumulateOutputs(outputClass, futureStream, reducer, empty);
        } catch (Throwable throwable) {
            return CompletableFuture.completedFuture(
                    MetabaseResponses.exceptionResponse(outputClass, throwable, logger)
            );
        }
    }

    private <Output> CompletableFuture<Output> accumulateOutputs(
            Class<Output> outputClass,
            Stream<CompletableFuture<Output>> futureStream,
            BinaryOperator<Output> reducer,
            Supplier<Output> empty) {
        try {
            var futures = futureStream.collect(toList());
            if (futures.isEmpty()) {
                return CompletableFuture.completedFuture(empty.get());
            }
            var result = CompletableFutures.accumulateWithCancellation(
                    futures,
                    MetabaseResponses.statusIsOk(outputClass),
                    reducer,
                    throwable -> MetabaseResponses.exceptionResponse(outputClass, throwable, logger),
                    responseHandlerExecutorService);
            return result.thenApply(output -> output == null ? empty.get() : output);
        } catch (Throwable throwable) {
            return CompletableFuture.completedFuture(
                    MetabaseResponses.exceptionResponse(outputClass, throwable, logger)
            );
        }
    }

}
