package ru.yandex.metabase.client.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillCloseWhenClosed;
import javax.annotation.WillNotClose;

import com.google.common.collect.ImmutableSet;
import com.google.protobuf.ByteString;
import com.google.protobuf.UnsafeByteOperations;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;
import io.netty.util.concurrent.DefaultThreadFactory;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.monlib.metrics.labels.validate.LabelValidationFilter;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsResponse;
import ru.yandex.solomon.metabase.api.protobuf.TSliceOptions;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsResponse;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.solomon.slog.ResolvedLogMetricsBuilderImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaRecord;
import ru.yandex.solomon.util.labelStats.LabelStatsCollectors;
import ru.yandex.solomon.util.labelStats.LabelStatsConverter;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;
import ru.yandex.solomon.util.protobuf.ByteStrings;

import static java.util.Objects.requireNonNull;
import static ru.yandex.solomon.labels.protobuf.LabelValidationFilterConverter.protoToFilter;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class MetabaseClientStub extends AbstractMetabaseClient {
    private final ConcurrentMap<Integer, Shard> shardsByNumId = new ConcurrentHashMap<>();
    @Nonnull
    private volatile EMetabaseStatusCode predefinedStatusCode = EMetabaseStatusCode.OK;
    private volatile AvailabilityStatus availability = AvailabilityStatus.AVAILABLE;

    @WillCloseWhenClosed
    private ExecutorService executorService;
    @WillNotClose
    private ExecutorService responseHandleExecutorService;

    public MetabaseClientStub(@WillNotClose ExecutorService responseHandleExecutorService) {
        this.responseHandleExecutorService = responseHandleExecutorService;
        this.executorService = Executors.newFixedThreadPool(2,
                new DefaultThreadFactory("server-side-metabase-thread-pool")
        );
    }

    public void addMetrics(ru.yandex.solomon.metabase.api.protobuf.Metric... metrics) {
        for (var proto : metrics) {
            Metric metric = Metric.of(proto);
            var shard = getOrCreateShard(metric.shardKey);
            var prev = shard.metrics.put(metric.labels, metric);
            if (prev != null) {
                throw new IllegalStateException("Not unique labels: "+ proto);
            }
        }
    }

    public static ru.yandex.solomon.metabase.api.protobuf.Metric metric(MetricType type, Labels labels) {
        return ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
            .setMetricId(MetricId.newBuilder()
                .setShardId(ThreadLocalRandom.current().nextInt(1, 4097))
                .setLocalId(ThreadLocalRandom.current().nextLong())
                .build())
            .setType(MetricTypeConverter.toProto(type))
            .setCreatedAtMillis(System.currentTimeMillis())
            .addAllLabels(LabelConverter.labelsToProtoList(withoutShardKey(labels).addAll(shardKey(labels))))
            .build();
    }

    public void removeMetrics(Selectors selector) {
        var shardSelector = ShardSelectors.onlyShardKey(selector);
        var otherSelectors = ShardSelectors.withoutShardKey(selector);
        this.shardsByNumId.values()
            .stream()
            .filter(shard -> shardSelector.match(shard.shardKey))
            .forEach(shard -> {
                shard.metrics.values().removeIf(metric -> otherSelectors.match(metric.labels));
            });
    }

    public void predefineStatusCode(EMetabaseStatusCode statusCode) {
        this.predefinedStatusCode = statusCode;
    }

    public void setAvailability(AvailabilityStatus availability) {
        this.availability = availability;
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        return doRpcEmulatedOpAsync(() -> rpcEmulatedSearch(request));
    }

    @Override
    public CompletableFuture<TUniqueLabelsResponse> uniqueLabels(TUniqueLabelsRequest request) {
        return doRpcEmulatedOpAsync(() -> rpcEmulatedUniqueLabels(request));
    }

    @Override
    public CompletableFuture<TLabelNamesResponse> labelNames(TLabelNamesRequest request) {
        return doRpcEmulatedOpAsync(() -> rpcEmulatedLabelNames(request));
    }

    @Override
    public CompletableFuture<TLabelValuesResponse> labelValues(TLabelValuesRequest request) {
        return doRpcEmulatedOpAsync(() -> rpcEmulatedLabelValues(request));
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        return doRpcEmulatedOpAsync(() -> rpcEmulatedResolveOne(request));
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        return doRpcEmulatedOpAsync(() -> rpcEmulatedResolveMany(request));
    }

    @Override
    public CompletableFuture<TResolveLogsResponse> resolveLogs(TResolveLogsRequest request) {
        return doRpcEmulatedOpAsync(() -> rpcEmulatedResolveLogs(request));
    }

    @Override
    public AvailabilityStatus getAvailability() {
        return availability;
    }

    @Override
    public Stream<Labels> shards(Selectors selectors) {
        return shardsByNumId.values()
                .stream()
                .map(shard -> shard.shardKey)
                .filter(selectors::match);
    }

    private <T> CompletableFuture<T> doRpcEmulatedOpAsync(Supplier<T> supplier) {
        CompletableFuture<T> future = new CompletableFuture<>();
        return CompletableFuture.supplyAsync(supplier, executorService)
            .whenCompleteAsync((response, throwable) -> {
                if (throwable != null) {
                    future.completeExceptionally(throwable);
                } else {
                    future.complete(response);
                }
            }, responseHandleExecutorService);
    }

    @Override
    public void close() {
        executorService.shutdown();
    }

    private FindResponse rpcEmulatedSearch(FindRequest request) {
        EMetabaseStatusCode predefine = predefinedStatusCode;
        if (predefine != EMetabaseStatusCode.OK) {
            return FindResponse.newBuilder()
                .setStatus(predefine)
                .setStatusMessage("Predefined status code on stub")
                .build();
        }

        return search(request);
    }

    private TUniqueLabelsResponse rpcEmulatedUniqueLabels(TUniqueLabelsRequest request) {
        EMetabaseStatusCode predefine = predefinedStatusCode;
        if (predefine != EMetabaseStatusCode.OK) {
            return TUniqueLabelsResponse.newBuilder()
                .setStatus(predefine)
                .setStatusMessage("Predefined status code on stub")
                .build();
        }

        Selectors selector =
            LabelSelectorConverter.protoToSelectors(request.getSelectorsList());

        return TUniqueLabelsResponse.newBuilder()
            .setStatus(EMetabaseStatusCode.OK)
            .addAllLabelLists(matchedMetrics(selector)
                .map(s -> LabelConverter.labelsToProtoList(s.getFullLabels()))
                .map(labels -> labels.stream()
                    .filter(label -> request.getNamesList().contains(label.getKey()))
                    .collect(Collectors.toList()))
                .map(labels -> ru.yandex.solomon.model.protobuf.Labels.newBuilder()
                    .addAllLabels(labels)
                    .build())
                .collect(Collectors.toList()))
            .build();
    }

    private TLabelNamesResponse rpcEmulatedLabelNames(TLabelNamesRequest request) {
        EMetabaseStatusCode predefine = predefinedStatusCode;
        if (predefine != EMetabaseStatusCode.OK) {
            return TLabelNamesResponse.newBuilder()
                    .setStatus(predefine)
                    .setStatusMessage("Predefined status code on stub")
                    .build();
        }

        Selectors selector = LabelSelectorConverter.protoToSelectors(request.getSelectorsList());

        Set<String> unique = matchedMetrics(selector)
                .map(Metric::getFullLabels)
                .flatMap(Labels::stream)
                .map(Label::getKey)
                .collect(Collectors.toSet());

        return TLabelNamesResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .addAllNames(unique)
                .build();
    }

    private TLabelValuesResponse rpcEmulatedLabelValues(TLabelValuesRequest request) {
        EMetabaseStatusCode predefine = predefinedStatusCode;
        if (predefine != EMetabaseStatusCode.OK) {
            return TLabelValuesResponse.newBuilder()
                .setStatus(predefine)
                .setStatusMessage("Predefined status code on stub")
                .build();
        }

        return labelValuesImpl(request);
    }

    private ResolveOneResponse rpcEmulatedResolveOne(ResolveOneRequest request) {
        EMetabaseStatusCode predefine = predefinedStatusCode;
        if (predefine != EMetabaseStatusCode.OK) {
            return ResolveOneResponse.newBuilder()
                .setStatus(predefine)
                .setStatusMessage("Predefined status code on stub")
                .build();
        }

        return resolveOneImpl(request);
    }

    private ResolveManyResponse rpcEmulatedResolveMany(ResolveManyRequest request) {
        EMetabaseStatusCode predefine = predefinedStatusCode;
        if (predefine != EMetabaseStatusCode.OK) {
            return ResolveManyResponse.newBuilder()
                .setStatus(predefine)
                .setStatusMessage("Predefined status code on stub")
                .build();
        }

        return resolveManyImpl(request);
    }

    private TResolveLogsResponse rpcEmulatedResolveLogs(TResolveLogsRequest request) {
        EMetabaseStatusCode predefine = predefinedStatusCode;
        if (predefine != EMetabaseStatusCode.OK) {
            return TResolveLogsResponse.newBuilder()
                .setStatus(predefine)
                .setStatusMessage("Predefined status code on stub")
                .build();
        }

        return resolveLogsImpl(request);
    }

    private FindResponse search(FindRequest request) {
        TSliceOptions sliceOptions = request.getSliceOptions();
        int offset = sliceOptions.getOffset();
        int limit = sliceOptions.getLimit();

        List<Metric> matchedMetrics;

        if (request.getFillMetricName()) {
            Selectors selectors = LabelSelectorConverter.protoToSelectors(request.getNewSelectors());
            selectors = toInternalSelector(selectors);
                matchedMetrics = matchedMetrics(selectors)
                        .map(Metric::toNamedMetric)
                        .collect(Collectors.toList());
        } else {
            Selectors selectors = LabelSelectorConverter.protoToSelectors(request.getSelectorsList());
            matchedMetrics = matchedMetrics(selectors).collect(Collectors.toList());
        }

        var result = sliceMetrics(matchedMetrics, offset, limit);

        return FindResponse.newBuilder()
            .setStatus(EMetabaseStatusCode.OK)
            .addAllMetrics(result.stream()
                .map(Metric::toProto)
                .collect(Collectors.toList()))
            .setTotalCount(matchedMetrics.size())
            .build();
    }

    private static Selectors toInternalSelector(Selectors selectors) {
        if (selectors.hasKey(LabelKeys.SENSOR)) {
            throw new IllegalArgumentException("unexpected metric name label in selectors");
        }
        if (selectors.getNameSelector().isEmpty()) {
            return selectors;
        }
        return selectors.toBuilder().add(LabelKeys.SENSOR, selectors.getNameSelector()).build();
    }

    private <T> List<T> sliceMetrics(List<T> metrics, int offset, int limit) {
        int finishPos;

        int size = metrics.size();

        if (limit > 0) {
            try {
                finishPos = Math.min(size, Math.addExact(limit, offset));
            } catch(ArithmeticException e) {
                finishPos = size;
            }
        } else {
            finishPos = size;
        }

        return metrics.subList(offset, finishPos);
    }

    private TLabelValuesResponse labelValuesImpl(TLabelValuesRequest request) {
        Selectors selector = LabelSelectorConverter.protoToSelectors(request.getSelectorsList());
        LabelValidationFilter validationFilter = protoToFilter(request.getValidationFilter());
        Set<String> requestedNames = ImmutableSet.copyOf(request.getLabelsList());
        LabelValuesStats result = matchedMetrics(selector)
                .map(Metric::getFullLabels)
                .collect(LabelStatsCollectors.toLabelValuesStats(requestedNames));
        result.filter(request.getTextSearch());
        result.filter(validationFilter);
        result.limit(request.getLimit());
        return LabelStatsConverter.toProto(result);
    }

    private ResolveOneResponse resolveOneImpl(ResolveOneRequest request) {
        Labels labels = LabelConverter.protoToLabels(request.getLabelsList());
        var shardKey = shardKey(labels);
        var shard = getOrCreateShard(shardKey);
        var metric = shard.metrics.get(withoutShardKey(labels));

        ResolveOneResponse.Builder response = ResolveOneResponse.newBuilder();
        if (metric != null) {
            response = response.setStatus(EMetabaseStatusCode.OK).setMetric(metric.toProto());
        } else {
            response = response.setStatus(EMetabaseStatusCode.NOT_FOUND).setStatusMessage("Not found");
        }
        return response.build();
    }

    private ResolveManyResponse resolveManyImpl(ResolveManyRequest request) {
        List<Labels> labelLists = request.getListLabelsList().stream()
            .map(labelList -> Labels.builder()
                .addAll(LabelConverter.protoToLabels(labelList))
                .addAll(LabelConverter.protoToLabels(request.getCommonLabelsList()))
                .build())
            .distinct()
            .collect(Collectors.toList());

        List<ru.yandex.solomon.metabase.api.protobuf.Metric> result = new ArrayList<>(labelLists.size());
        for (var labels : labelLists) {
            var shardKey = shardKey(labels);
            Shard shard = getOrCreateShard(shardKey);
            var metric = shard.metrics.get(withoutShardKey(labels));
            if (metric != null) {
                result.add(metric.toProto());
            }
        }

        return ResolveManyResponse.newBuilder()
            .setStatus(EMetabaseStatusCode.OK)
            .addAllMetrics(result)
            .build();
    }

    private TResolveLogsResponse resolveLogsImpl(TResolveLogsRequest request) {
        var shard = shardsByNumId.get(request.getNumId());
        if (shard == null) {
            return TResolveLogsResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.SHARD_NOT_FOUND)
                .setStatusMessage("shard unknown")
                .build();
        }

        var result = TResolveLogsResponse.newBuilder().setStatus(EMetabaseStatusCode.OK);
        for (var unresolved : request.getUnresolvedLogMetaList()) {
            result.addResolvedLogMetrics(resolveOneLog(shard, unresolved));
        }

        return result.build();
    }

    private ByteString resolveOneLog(Shard shard, ByteString data) {
        var record = new UnresolvedLogMetaRecord();
        var now = System.currentTimeMillis();
        try (
                var it = new UnresolvedLogMetaIteratorImpl(ByteStrings.toByteBuf(data));
                var builder = new ResolvedLogMetricsBuilderImpl(shard.numId, CompressionAlg.LZ4, ByteBufAllocator.DEFAULT);
        ) {
            while (it.next(record)) {
                var metric = shard.metrics.compute(record.labels, (labels, prev) -> {
                    if (prev == null) {
                        return new Metric(shard.shardKey, "", record.labels, record.type, nextShardId(), nextLocalId(), now);
                    }

                    if (prev.type != record.type) {
                        return new Metric(prev.shardKey, "", prev.labels, record.type, prev.shardId, prev.localId, prev.createdAt, prev.sequence);
                    }

                    return prev;
                });
                builder.onMetric(metric.type, metric.labels, metric.shardId, metric.localId);
            }
            var buffer = builder.build();
            var result = UnsafeByteOperations.unsafeWrap(ByteBufUtil.getBytes(buffer));
            buffer.release();
            return result;
        }
    }

    private int nextShardId() {
        return ThreadLocalRandom.current().nextInt(1, 4097);
    }

    private long nextLocalId() {
        return ThreadLocalRandom.current().nextLong();
    }

    private Stream<Metric> matchedMetrics(Selectors selectors) {
        var onlyShard = ShardSelectors.onlyShardKey(selectors);
        var other = ShardSelectors.withoutShardKey(selectors);
        return shardsByNumId.values().stream()
            .filter(shard -> onlyShard.match(shard.shardKey))
            .flatMap(shard -> shard.metrics.values().stream())
            .filter(metric -> other.match(metric.labels))
            .sorted();
    }

    private Shard getOrCreateShard(Labels key) {
        return shardsByNumId.computeIfAbsent(key.hashCode(), ignore -> new Shard(key));
    }

    private static Labels withoutShardKey(Labels labels) {
        return labels.removeByKey(LabelKeys.PROJECT)
            .removeByKey(LabelKeys.CLUSTER)
            .removeByKey(LabelKeys.SERVICE);
    }

    private static Labels shardKey(Labels labels) {
        LabelsBuilder builder = new LabelsBuilder(3);
        for (String key : List.of(LabelKeys.PROJECT, LabelKeys.CLUSTER, LabelKeys.SERVICE)) {
            var label = labels.findByKey(key);
            if (label != null) {
                builder.add(label);
            }
        }
        return builder.build();
    }

    private static class Shard {
        private final int numId;
        private final Labels shardKey;
        private final ConcurrentMap<Labels, Metric> metrics;

        public Shard(Labels key) {
            this.numId = key.hashCode();
            this.shardKey = key;
            this.metrics = new ConcurrentHashMap<>();
        }
    }

    private static class Metric implements Comparable<Metric> {
        private static final AtomicLong SEQ = new AtomicLong();
        private final Labels shardKey;
        private final String name;
        private final Labels labels;
        private final MetricType type;
        private final int shardId;
        private final long localId;
        private final long createdAt;
        private final long sequence;

        public Metric(Labels shardKey, String name, Labels labels, MetricType type, int shardId, long localId, long createdAt) {
            this(shardKey, name, labels, type, shardId, localId, createdAt, SEQ.getAndIncrement());
        }

        public Metric(Labels shardKey, String name, Labels labels, MetricType type, int shardId, long localId, long createdAt, long seq) {
            this.shardKey = shardKey;
            this.name = name;
            this.labels = labels;
            this.type = type;
            this.shardId = shardId;
            this.localId = localId;
            this.createdAt = createdAt;
            this.sequence = seq;
        }

        public static Metric of(ru.yandex.solomon.metabase.api.protobuf.Metric metric) {
            int shardId = metric.getMetricId().getShardId();
            long localId = metric.getMetricId().getLocalId();
            var labels = LabelConverter.protoToLabels(metric.getLabelsList());
            var shardKey = shardKey(labels);
            var type = requireNonNull(MetricTypeConverter.fromProto(metric.getType()));
            var createdAt = metric.getCreatedAtMillis();
            return new Metric(shardKey, metric.getName(), withoutShardKey(labels), type, shardId, localId, createdAt);
        }

        private Metric toNamedMetric() {
            Label metricNameLabel = labels.findByKey(LabelKeys.SENSOR);
            if (metricNameLabel == null) {
                throw new IllegalStateException("metric name label is required, but: " + labels);
            }
            String metricName = metricNameLabel.getValue();
            Labels labelsWithoutMetricName = labels.removeByKey(LabelKeys.SENSOR);
            return new Metric(shardKey, metricName, labelsWithoutMetricName, type, shardId, localId, createdAt, sequence);
        }

        private ru.yandex.solomon.metabase.api.protobuf.Metric toProto() {
            return ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                .setMetricId(MetricId.newBuilder()
                    .setShardId(shardId)
                    .setLocalId(localId)
                    .build())
                .setType(requireNonNull(MetricTypeConverter.toProto(type)))
                .setName(name)
                .addAllLabels(LabelConverter.labelsToProtoList(getFullLabels()))
                .setCreatedAtMillis(createdAt)
                .build();
        }

        public Labels getFullLabels() {
            return labels.addAll(shardKey);
        }

        @Override
        public int compareTo(Metric o) {
            return Long.compare(sequence, o.sequence);
        }
    }
}
