package ru.yandex.metabase.client.impl;

import java.nio.channels.ClosedChannelException;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillCloseWhenClosed;

import com.google.common.base.Throwables;
import com.google.common.net.HostAndPort;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;

import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.metabase.MetabaseServiceGrpc;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindResponse;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsResponse;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusRequest;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusResponse;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsResponse;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
class MetabaseNodeClient implements AutoCloseable {
    @WillCloseWhenClosed
    private final GrpcTransport transport;
    private final MetricRegistry registry;

    MetabaseNodeClient(HostAndPort address, MetabaseClientOptions options) {
        this.transport = new GrpcTransport(address, options.getGrpcOptions());
        this.registry = options.getGrpcOptions().getMetricRegistry();
    }

    CompletableFuture<TServerStatusResponse> serverStatus(TServerStatusRequest request) {
        return unaryCall(MetabaseServiceGrpc.getServerStatusMethod(),
                request,
                request.getDeadlineMillis(),
                TServerStatusResponse::getStatus,
                status -> TServerStatusResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<CreateOneResponse> createOne(CreateOneRequest request) {
        return unaryCall(MetabaseServiceGrpc.getCreateOneMethod(),
                request,
                request.getDeadlineMillis(),
                CreateOneResponse::getStatus,
                status -> CreateOneResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<CreateManyResponse> createMany(CreateManyRequest request) {
        return unaryCall(MetabaseServiceGrpc.getCreateManyMethod(),
                request,
                request.getDeadlineMillis(),
                CreateManyResponse::getStatus,
                status -> CreateManyResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        return unaryCall(MetabaseServiceGrpc.getResolveOneMethod(),
                request,
                request.getDeadlineMillis(),
                ResolveOneResponse::getStatus,
                status -> ResolveOneResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        return unaryCall(MetabaseServiceGrpc.getResolveManyMethod(),
                request,
                request.getDeadlineMillis(),
                ResolveManyResponse::getStatus,
                status -> ResolveManyResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<DeleteManyResponse> deleteMany(DeleteManyRequest request) {
        return unaryCall(MetabaseServiceGrpc.getDeleteManyMethod(),
                request,
                request.getDeadlineMillis(),
                DeleteManyResponse::getStatus,
                status -> DeleteManyResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<FindResponse> find(FindRequest request) {
        return unaryCall(MetabaseServiceGrpc.getFindMethod(),
                request,
                request.getDeadlineMillis(),
                FindResponse::getStatus,
                status -> FindResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request) {
        return unaryCall(MetabaseServiceGrpc.getMetricNamesMethod(),
                request,
                request.getDeadlineMillis(),
                MetricNamesResponse::getStatus,
                status -> MetricNamesResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<TLabelValuesResponse> labelValues(TLabelValuesRequest request) {
        return unaryCall(MetabaseServiceGrpc.getLabelValuesMethod(),
                request,
                request.getDeadlineMillis(),
                TLabelValuesResponse::getStatus,
                status -> TLabelValuesResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<TLabelNamesResponse> labelNames(TLabelNamesRequest request) {
        return unaryCall(MetabaseServiceGrpc.getLabelNamesMethod(),
                request,
                request.getDeadlineMillis(),
                TLabelNamesResponse::getStatus,
                status -> TLabelNamesResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<TUniqueLabelsResponse> uniqueLabels(TUniqueLabelsRequest request) {
        return unaryCall(MetabaseServiceGrpc.getUniqueLabelsMethod(),
                request,
                request.getDeadlineMillis(),
                TUniqueLabelsResponse::getStatus,
                status -> TUniqueLabelsResponse.newBuilder()
                        .setStatus(status.getCode())
                        .setStatusMessage(status.getDetails())
                        .build());
    }

    CompletableFuture<TResolveLogsResponse> resolveLogs(TResolveLogsRequest request) {
        return unaryCall(MetabaseServiceGrpc.getResolveLogsMethod(),
            request,
            request.getDeadlineMillis(),
            TResolveLogsResponse::getStatus,
            status -> TResolveLogsResponse.newBuilder()
                .setStatus(status.getCode())
                .setStatusMessage(status.getDetails())
                .build());
    }

    private <ReqT, RespT> CompletableFuture<RespT> unaryCall(
            MethodDescriptor<ReqT, RespT> method,
            ReqT request,
            long deadline,
            Function<RespT, EMetabaseStatusCode> statusCodeFn,
            Function<RequestStatus, RespT> errorHandling)
    {
        return transport.unaryCall(method, request, deadline).handle((response, throwable) -> {
            EMetabaseStatusCode statusCode;
            if (throwable != null) {
                RequestStatus status = classifyError(throwable);
                statusCode = status.getCode();
                response = errorHandling.apply(status);
            } else {
                statusCode = statusCodeFn.apply(response);
            }

            Labels labels = Labels.of("endpoint", method.getFullMethodName(), "code", statusCode.name());
            registry.rate("metabase.client.call.status", labels).inc();
            return response;
        });
    }

    private RequestStatus classifyError(Throwable error) {
        final Throwable unwrapped = CompletableFutures.unwrapCompletionException(error);
        if (!(unwrapped instanceof StatusRuntimeException)) {
            return new RequestStatus(EMetabaseStatusCode.INTERNAL_ERROR, Throwables.getStackTraceAsString(unwrapped));
        }

        StatusRuntimeException errorToClassify = (StatusRuntimeException) unwrapped;
        Status status = errorToClassify.getStatus();
        switch (errorToClassify.getStatus().getCode()) {
            case RESOURCE_EXHAUSTED:
                return new RequestStatus(EMetabaseStatusCode.RESOURCE_EXHAUSTED, status.toString());
            case DEADLINE_EXCEEDED:
                return new RequestStatus(EMetabaseStatusCode.DEADLINE_EXCEEDED, status.toString());
            case UNAVAILABLE:
            case CANCELLED:
                return new RequestStatus(EMetabaseStatusCode.NODE_UNAVAILABLE, status.toString());
            case UNKNOWN:
                if (errorToClassify.getCause() instanceof ClosedChannelException) {
                    return new RequestStatus(EMetabaseStatusCode.NODE_UNAVAILABLE, status.toString());
                }
            default:
                return new RequestStatus(EMetabaseStatusCode.INTERNAL_ERROR, status.toString());
        }
    }

    @Override
    public void close() {
        transport.close();
    }

    public HostAndPort getAddress() {
        return transport.getAddress();
    }

    @Override
    public String toString() {
        return "MetabaseNodeClient{" +
                "address=" + transport.getAddress() +
                '}';
    }
}
