package ru.yandex.metabase.client.impl;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.WillCloseWhenClosed;

import com.google.common.net.HostAndPort;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusRequest;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.solomon.util.actors.PingActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
class MetabaseNodeImpl implements MetabaseNode {
    private static final Logger logger = LoggerFactory.getLogger(MetabaseNodeImpl.class);

    @WillCloseWhenClosed
    private final MetabaseNodeClient client;
    private final PingActorRunner actor;
    private volatile ShardsState state;

    public MetabaseNodeImpl(HostAndPort address, MetabaseClientOptions opts, Executor executor, ScheduledExecutorService timer) {
        this.client = new MetabaseNodeClient(address, opts);
        this.state = ShardsState.init(getFqdn());
        this.actor = PingActorRunner.newBuilder()
            .operation("refresh_metabase_shards_state")
            .pingInterval(Duration.ofSeconds(5))
            .backoffMaxDelay(Duration.ofSeconds(30))
            .timer(timer)
            .executor(executor)
            .onPing(attempt -> forceUpdate())
            .build();
        actor.forcePing();
    }

    @Override
    public ShardsState getState() {
        return state;
    }

    public AvailabilityStatus getAvailability() {
        return state.getAvailability();
    }

    @Override
    public MetabaseNodeClient getClient() {
        return client;
    }

    @Override
    public CompletableFuture<?> forceUpdate() {
        var oldState = state;
        var request = TServerStatusRequest.newBuilder()
            .setShardIdsHash(oldState.getHash())
            .build();

        return client.serverStatus(request)
            .whenComplete((response, e) -> {
                if (e != null) {
                    logger.warn("Metabase#ServerStatus {} failed, reason: {}", getFqdn(), e);
                    return;
                }
                if (response.getStatus() != EMetabaseStatusCode.OK) {
                    logger.warn("Metabase#ServerStatus {} failed, reason: {}: {}", getFqdn(), response.getStatus(), response.getStatusMessage());
                    return;
                }
                this.state = ShardsState.update(getFqdn(), oldState, response);
            });
    }

    public String getFqdn() {
        return client.getAddress().getHost();
    }

    @Override
    public void close() {
        actor.close();
        client.close();
    }
}
