package ru.yandex.metabase.client.impl;

import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.metabase.api.protobuf.CreateManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusRequest;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;

/**
 * Signs all requests with generationIdToPartitionId except {@link TServerStatusRequest}
 *
 * @author Egor Litvinenko
 */
@ParametersAreNonnullByDefault
class MetabaseRequests {

    @FunctionalInterface
    private interface AddFullPartitionKey<T> {
        T apply(T request, PartitionedShard shard, int partitionId);
    }

    private static final Map<Class<?>, AddFullPartitionKey<?>> FULL_PARTITION_KEY_BUILDERS_MAP =
            Map.ofEntries(
                    Map.entry(ResolveManyRequest.class,
                            (requestBuilder, shard, partitionId) -> ((ResolveManyRequest.Builder) requestBuilder).setPartitionId(partitionId).setGenerationId(shard.getGenerationId())),
                    Map.entry(ResolveOneRequest.class,
                            (requestBuilder, shard, partitionId) -> ((ResolveOneRequest.Builder) requestBuilder).setPartitionId(partitionId).setGenerationId(shard.getGenerationId())),
                    Map.entry(CreateManyRequest.class,
                            (requestBuilder, shard, partitionId) -> ((CreateManyRequest.Builder) requestBuilder).setPartitionId(partitionId).setGenerationId(shard.getGenerationId())),
                    Map.entry(CreateOneRequest.class,
                            (requestBuilder, shard, partitionId) -> ((CreateOneRequest.Builder) requestBuilder).setPartitionId(partitionId).setGenerationId(shard.getGenerationId())),
                    Map.entry(FindRequest.class,
                            (requestBuilder, shard, partitionId) -> ((FindRequest.Builder) requestBuilder).setShardId(shard.getNumId()).setPartitionId(partitionId).setGenerationId(shard.getGenerationId())),
                    Map.entry(DeleteManyRequest.class,
                            (requestBuilder, shard, partitionId) -> ((DeleteManyRequest.Builder) requestBuilder).setPartitionId(partitionId).setGenerationId(shard.getGenerationId())),
                    Map.entry(TResolveLogsRequest.class,
                            (requestBuilder, shard, partitionId) -> ((TResolveLogsRequest.Builder) requestBuilder).setNumId(shard.getNumId()).setPartitionId(partitionId).setGenerationId(shard.getGenerationId())),
                    Map.entry(MetricNamesRequest.class,
                            (requestBuilder, shard, partitionId) -> ((MetricNamesRequest.Builder) requestBuilder).setShardId(shard.getNumId()).setPartitionId(partitionId).setGenerationId(shard.getGenerationId())),
                    Map.entry(TLabelValuesRequest.class,
                            (requestBuilder, shard, partitionId) -> ((TLabelValuesRequest.Builder) requestBuilder).setShardId(shard.getNumId()).setPartitionId(partitionId).setGenerationId(shard.getGenerationId())),
                    Map.entry(TLabelNamesRequest.class,
                            (requestBuilder, shard, partitionId) -> ((TLabelNamesRequest.Builder) requestBuilder).setShardId(shard.getNumId()).setPartitionId(partitionId).setGenerationId(shard.getGenerationId())),
                    Map.entry(TUniqueLabelsRequest.class,
                            (requestBuilder, shard, partitionId) -> ((TUniqueLabelsRequest.Builder) requestBuilder).setShardId(shard.getNumId()).setPartitionId(partitionId).setGenerationId(shard.getGenerationId()))
            );

    static <TBuilder, TRequest> TBuilder builderWithPartitionKey(Class<TRequest> clazz, TBuilder request, PartitionedShard shard, int partitionId) {
        return ( (AddFullPartitionKey<TBuilder>) FULL_PARTITION_KEY_BUILDERS_MAP.get(clazz)).apply(request, shard, partitionId);
    }

    private static final Map<Class<?>, AddFullPartitionKey<?>> FULL_PARTITION_KEY_SETTER_MAP =
            Map.ofEntries(
                    Map.entry(ResolveManyRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(ResolveManyRequest.class, ((ResolveManyRequest) request).toBuilder(), shard, partitionId).build()),
                    Map.entry(ResolveOneRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(ResolveOneRequest.class, ((ResolveOneRequest) request).toBuilder(), shard, partitionId).build()),
                    Map.entry(CreateManyRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(CreateManyRequest.class, ((CreateManyRequest) request).toBuilder(), shard, partitionId).build()),
                    Map.entry(CreateOneRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(CreateOneRequest.class, ((CreateOneRequest) request).toBuilder(), shard, partitionId).build()),
                    Map.entry(FindRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(FindRequest.class, ((FindRequest) request).toBuilder(), shard, partitionId).build()),
                    Map.entry(DeleteManyRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(DeleteManyRequest.class, ((DeleteManyRequest) request).toBuilder(), shard, partitionId).build()),
                    Map.entry(TResolveLogsRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(TResolveLogsRequest.class, ((TResolveLogsRequest) request).toBuilder(), shard, partitionId).build()),
                    Map.entry(MetricNamesRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(MetricNamesRequest.class, ((MetricNamesRequest) request).toBuilder(), shard, partitionId).build()),
                    Map.entry(TLabelValuesRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(TLabelValuesRequest.class, ((TLabelValuesRequest) request).toBuilder(), shard, partitionId).build()),
                    Map.entry(TLabelNamesRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(TLabelNamesRequest.class, ((TLabelNamesRequest) request).toBuilder(), shard, partitionId).build()),
                    Map.entry(TUniqueLabelsRequest.class,
                            (request, shard, partitionId) -> builderWithPartitionKey(TUniqueLabelsRequest.class, ((TUniqueLabelsRequest) request).toBuilder(), shard, partitionId).build())
            );

    static <T> T withPartitionKey(Class<T> clazz, T request, PartitionedShard shard, int partitionId) {
        return ( (AddFullPartitionKey<T>) FULL_PARTITION_KEY_SETTER_MAP.get(clazz)).apply(request, shard, partitionId);
    }

}
