package ru.yandex.metabase.client.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import io.grpc.Status;

import ru.yandex.metabase.client.MetabasePartitions;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.model.protobuf.Label;

/**
 * @author Egor Litvinenko
 * */
public class PartitionedShardResolver {

    private final MetabaseCluster.State state;

    public static PartitionedShardResolver of(MetabaseCluster cluster) {
        return new PartitionedShardResolver(cluster);
    }

    public PartitionedShardResolver(MetabaseCluster metabaseCluster) {
        this.state = metabaseCluster.getState();
    }

    public Stream<? extends PartitionedShard> allShards(Selectors selectors) {
        if (selectors.isEmpty()) {
            return state.allShards();
        }
        return resolve(selectors, false);
    }

    public Stream<? extends PartitionedShard> resolve(Selectors selectors) {
        return resolve(selectors, true);
    }

    private Stream<? extends PartitionedShard> resolve(Selectors selectors, boolean onlyReady) {
        var shardKey = ShardSelectors.getShardKeyOrNull(selectors);
        if (shardKey != null) {
            return resolveByShardKey(shardKey, onlyReady);
        }
        var project = selectors.findByKey(LabelKeys.PROJECT);
        if (project != null && project.isExact()) {
            var result = state.projectShards(MyInterners.pcs().intern(project.getValue())).stream();
            return allOrReady(result, onlyReady);
        }
        var shardSelectors = ShardSelectors.onlyShardKey(selectors);
        if (shardSelectors.isEmpty()) {
            return allOrReady(state.allShards(), onlyReady);
        } else {
            return allOrReady(state.allShards(), onlyReady)
                    .filter(shard -> shardSelectors.match(shard.getKey()));
        }
    }

    public Stream<? extends PartitionedShard> resolveByShardKey(ShardKey shardKey, boolean onlyReady) {
        return resolveByNumId(state.shardNumId(shardKey), onlyReady);
    }

    public Stream<? extends PartitionedShard> resolveByNumId(int numId, boolean onlyReady) {
        var shard = state.shardByNumId(numId);
        if (shard != null) {
            if (onlyReady) {
                validateShard(shard);
            }
            return Stream.of(shard);
        }
        return Stream.of();
    }

    public Stream<? extends PartitionedShard> resolveByLabels(List<Label> labels) {
        return resolve(labels.stream().map(label -> Selector.exact(label.getKey(), label.getValue())).collect(Selectors.collector()));
    }

    public Stream<PartitionKey> resolvePartitionByLabels(List<Label> labels, boolean failOnNotFound) {
        var project = "";
        var cluster = "";
        var service = "";
        List<Label> labelsWithoutPcs = new ArrayList<>(Math.max(1, labels.size() - 3));
        for (Label label : labels) {
            switch (label.getKey()) {
                case LabelKeys.PROJECT: project = label.getValue(); break;
                case LabelKeys.CLUSTER: cluster = label.getValue(); break;
                case LabelKeys.SERVICE: service = label.getValue(); break;
                default: labelsWithoutPcs.add(label); break;
            }
        }
        var noShardKey = project.isEmpty() || cluster.isEmpty() || service.isEmpty();
        if (noShardKey) {
            throw MetabaseResponses.createException("Shard key is not defined: " + labels, EMetabaseStatusCode.INVALID_REQUEST);
        }
        var shardKey = MyInterners.shardKey().intern(ShardKey.create(
                MyInterners.pcs().intern(project),
                MyInterners.pcs().intern(cluster),
                MyInterners.pcs().intern(service)
        ));
        var shard = state.shardByShardKey(shardKey);
        if (shard == null) {
            if (failOnNotFound) {
                throw MetabaseResponses.createException("Not found shard for metric: " + labels, EMetabaseStatusCode.SHARD_NOT_FOUND);
            }
            return Stream.of();
        }
        validateShard(shard);
        labelsWithoutPcs.sort(MetabasePartitions.labelHashComparator());
        var partitionId = MetabasePartitions.labelsPartition(shard.getTotalPartitions(), labelsWithoutPcs);
        return Stream.of(shard.withOnePartition(partitionId));
    }

    private boolean validateShard(MetabaseShard shard) {
        if (shard.isReady()) {
            return true;
        } else {
            throw Status.UNAVAILABLE.withDescription("Shard " + shard.getKey() + " is not available yet").asRuntimeException();
        }
    }

    private static Stream<MetabaseShard> allOrReady(Stream<MetabaseShard> shards, boolean onlyReady) {
        return onlyReady ? shards.filter(MetabaseShard::isReady) : shards;
    }

}
