package ru.yandex.metabase.client.impl;

import java.util.List;
import java.util.OptionalInt;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2LongMap;
import it.unimi.dsi.fastutil.objects.Object2LongOpenHashMap;

import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusResponse;
import ru.yandex.solomon.metabase.api.protobuf.TShardStatus;
import ru.yandex.solomon.model.protobuf.Label;
import ru.yandex.solomon.selfmon.AvailabilityStatus;

/**
 * @author Vladimir Gordiychuk
 */
final class ShardsState {
    private final String node;
    private final long hash;
    private final MetabaseShard[] shards;
    private final long createdAt;
    private final AvailabilityStatus availability;
    private final int readyPartitionsCount;
    private final OptionalInt totalPartitionCount;
    private final int inactivePartitionCount;

    private ShardsState(
            String node,
            long hash,
            MetabaseShard[] shards,
            AvailabilityStatus availability,
            int readyPartitionsCount,
            OptionalInt totalPartitionCount,
            int inactivePartitionCount,
            long createdAt)
    {
        this.node = node;
        this.hash = hash;
        this.shards = shards;
        this.readyPartitionsCount = readyPartitionsCount;
        this.totalPartitionCount = totalPartitionCount;
        this.inactivePartitionCount = inactivePartitionCount;
        this.createdAt = createdAt;
        this.availability = availability;
    }

    public static ShardsState init(String node) {
        return new ShardsState(node, 0, new MetabaseShard[0],
                new AvailabilityStatus(0, node + " not initialized yet"),
                0, OptionalInt.empty(), 0, 0);
    }

    public static ShardsState update(String node, ShardsState oldState, TServerStatusResponse response) {
        long hash = oldState.hash;
        MetabaseShard[] metabaseShards = oldState.shards;
        AvailabilityStatus availability = oldState.availability;
        int readyPartitionsCount = oldState.readyPartitionsCount;
        OptionalInt totalPartitionCount = oldState.totalPartitionCount;
        int inactivePartitionCount = oldState.inactivePartitionCount;
        long createdAt = oldState.getCreatedAt();

        boolean hasDiff = !node.equals(oldState.node);

        if (hash == 0 || hash != response.getShardIdsHash()) {
            final long createdAtNew = System.nanoTime();

            final Object2LongMap<ShardKey> maxGenerationId = response.getPartitionStatusList().stream()
                    .collect(Collectors.toMap(
                            status -> createKey(status.getLabelsList()), TShardStatus::getGenerationId, Long::max, Object2LongOpenHashMap::new));

            readyPartitionsCount = 0;
            inactivePartitionCount = response.getInactivePartitionCount();

            Int2ObjectMap<MetabaseShard> shardsNew = new Int2ObjectOpenHashMap<>(oldState.shards.length);
            for (TShardStatus partitionStatus : response.getPartitionStatusList()) {
                var key = createKey(partitionStatus.getLabelsList());
                var maxGenId = maxGenerationId.getOrDefault(key, 0l);
                if (maxGenId > partitionStatus.getGenerationId()) {
                    inactivePartitionCount++;
                    // LB updates generationId, they are eventually the same, ignore not updated partitions
                    continue;
                }
                int numId = partitionStatus.getNumId();
                var shardCandidate = new MetabaseShard(
                        createdAtNew,
                        key,
                        numId,
                        maxGenId,
                        Math.max(partitionStatus.getTotalPartitions(), 1)
                );
                var shard = shardsNew.merge(numId, shardCandidate, MetabaseShard::mergeShardInfoFromServers);
                boolean allowNew = partitionStatus.getMetricCount() == 0 || partitionStatus.getMetricCount() < partitionStatus.getMetricLimit();
                shard.addPartition(maxGenId, partitionStatus.getPartitionId(), MyInterners.fqdn().intern(node), partitionStatus.getReady(), allowNew);
                if (partitionStatus.getReady()) {
                    readyPartitionsCount++;
                }
            }
            metabaseShards = shardsNew.values().toArray(new MetabaseShard[shardsNew.size()]);
            hash = response.getShardIdsHash();
            availability = calculateAvailability(node, shardsNew.values().stream().mapToInt(MetabaseShard::getDiscoveredPartitions).sum(), readyPartitionsCount);
            createdAt = createdAtNew;
            hasDiff = true;
        }

        OptionalInt newTotalPartitionCount = response.getTotalPartitionCountKnown() ?
                OptionalInt.of(response.getTotalPartitionCount()) : OptionalInt.empty();

        if (!newTotalPartitionCount.equals(totalPartitionCount)) {
            totalPartitionCount = newTotalPartitionCount;
            hasDiff = true;
        }

        if (oldState.inactivePartitionCount != inactivePartitionCount) {
            hasDiff = true;
        }

        if (!hasDiff) {
            return oldState;
        }

        return new ShardsState(
                node,
                hash,
                metabaseShards,
                availability,
                readyPartitionsCount,
                totalPartitionCount,
                inactivePartitionCount,
                createdAt
        );
    }

    private static AvailabilityStatus calculateAvailability(String node, int total, int readyCount) {
        if (readyCount >= total) {
            return AvailabilityStatus.AVAILABLE;
        }

        String details = node + ": ready shards " + readyCount + "/" + total;
        double availability = (double) readyCount / (double) total;
        return new AvailabilityStatus(availability, details);
    }

    public AvailabilityStatus getAvailability() {
        return availability;
    }

    public OptionalInt getTotalPartitionCount() {
        return totalPartitionCount;
    }

    public int getInactivePartitionCount() {
        return inactivePartitionCount;
    }

    public Stream<MetabaseShard> getShards() {
        return Stream.of(shards);
    }

    public long getCreatedAt() {
        return createdAt;
    }

    public long getHash() {
        return hash;
    }

    private static ShardKey createKey(List<Label> labels) {
        String project = null;
        String cluster = null;
        String service = null;
        for (var label : labels) {
            switch (label.getKey()) {
                case LabelKeys.PROJECT:
                    project = MyInterners.pcs().intern(label.getValue());
                    break;
                case LabelKeys.CLUSTER:
                    cluster = MyInterners.pcs().intern(label.getValue());
                    break;
                case LabelKeys.SERVICE:
                    service = MyInterners.pcs().intern(label.getValue());
                    break;
            }
        }
        return MyInterners.shardKey().intern(ShardKey.create(project, cluster, service));
    }

    @Override
    public String toString() {
        return "ShardsState{" +
                "node='" + node + '\'' +
                ", metabasePartitions.length=" + shards.length +
                ", createdAt=" + createdAt +
                '}';
    }

}
