package ru.yandex.solomon.metrics.client;

import java.time.Instant;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.common.RequestProducer;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class AbstractRequest {
    private final long deadline;
    private final long softDeadline;
    @Nullable
    private final String destination;
    @Nonnull
    private final RequestProducer producer;
    private final String subjectId;
    private final boolean dataProxyAllowed;

    public AbstractRequest(Builder builder) {
        this.deadline = builder.deadline;
        this.softDeadline = builder.softDeadline;
        this.destination = builder.destination;
        this.producer = builder.producer != null
            ? builder.producer
            : RequestProducer.REQUEST_PRODUCER_UNSPECIFIED;
        this.subjectId = builder.subjectId != null
            ? builder.subjectId
            : "UNSPECIFIED";
        this.dataProxyAllowed = builder.dataProxyAllowed;
    }

    /**
     * Hard deadline: if no data is available before deadline, fail
     * E.g. wait for the first DC response deadline
     */
    public long getDeadline() {
        return deadline;
    }

    /**
     * Soft deadline: if some (may be incomplete) data is available by the soft deadline, return it as is
     * E.g. wait for the second DC response deadline
     * May be zero, means unset
     */
    public long getSoftDeadline() {
        return softDeadline;
    }

    @Nullable
    public String getDestination() {
        return destination;
    }

    /**
     * Empty result indicate that force destinations not specified
     */
    public Set<String> getDestinations() {
        return destination == null ? Set.of() : Set.of(destination);
    }

    /**
     * Auth subject id
     */
    public String getSubjectId() {
        return subjectId;
    }

    @Nonnull
    public RequestProducer getProducer() {
        return producer;
    }

    public boolean isDataProxyAllowed() {
        return dataProxyAllowed;
    }

    @Override
    public String toString() {
        return "AbstractRequest{" +
            "deadline=" + deadline +
            (deadline != softDeadline ? ", softDeadline=" + softDeadline : "") +
            ", destination='" + destination + '\'' +
            '}';
    }

    @NotThreadSafe
    public static abstract class Builder<T extends Builder<T>> {
        private long deadline;
        private long softDeadline;
        @Nullable
        private String destination;
        @Nullable
        private RequestProducer producer;
        @Nullable
        private String subjectId;
        private boolean dataProxyAllowed = true;

        public Builder() {
        }

        public Builder(AbstractRequest request) {
            this.deadline = request.deadline;
            this.softDeadline = request.softDeadline;
            this.destination = request.destination;
            this.producer = request.producer;
            this.subjectId = request.subjectId;
        }

        public T setDeadline(long deadline) {
            this.deadline = deadline;
            return self();
        }

        public T setDeadline(Instant deadline) {
            this.deadline = deadline.toEpochMilli();
            return self();
        }

        public T setSoftDeadline(long softDeadline) {
            this.softDeadline = softDeadline;
            return self();
        }

        public T setSoftDeadline(Instant softDeadline) {
            this.softDeadline = softDeadline.toEpochMilli();
            return self();
        }

        public T setDestination(@Nullable String destination) {
            this.destination = destination == null ? null : StringUtils.trimToNull(destination);
            return self();
        }

        public T setProducer(@Nullable RequestProducer producer) {
            this.producer = producer;
            return self();
        }

        public T setSubjectId(@Nullable String subjectId) {
            this.subjectId = subjectId;
            return self();
        }

        public T forbidDataProxy() {
            this.dataProxyAllowed = false;
            return self();
        }

        public T allowDataProxy() {
            this.dataProxyAllowed = true;
            return self();
        }

        public T self() {
            return (T) this;
        }
    }
}
