package ru.yandex.solomon.metrics.client;

import java.util.List;

import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.metabase.api.protobuf.MetabaseMetricId;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TSliceOptions;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;
import ru.yandex.solomon.model.StockpileKey;
import ru.yandex.solomon.model.protobuf.Label;
import ru.yandex.solomon.model.protobuf.Labels;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.stockpile.api.TReadRequest;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class Converters {
    public static ru.yandex.solomon.metabase.api.protobuf.FindRequest toProto(FindRequest request) {
        var builder = ru.yandex.solomon.metabase.api.protobuf.FindRequest.newBuilder()
            .setDeadlineMillis(request.getDeadline())
            .setFolderId(request.getFolderId())
            .setSliceOptions(TSliceOptions.newBuilder()
                .setLimit(request.getLimit())
                .setOffset(request.getOffset()))
            .setFillMetricName(request.isUseNewFormat());

        if (request.isUseNewFormat()) {
            builder.setNewSelectors(LabelSelectorConverter.selectorsToNewProto(request.getSelectors()));
        } else {
            builder.addAllSelectors(LabelSelectorConverter.selectorsToProto(request.getSelectors()));
        }

        return builder.build();
    }

    public static ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest toProto(MetricNamesRequest request) {
        return ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest.newBuilder()
            .setFolderId(request.getFolderId())
            .setSelectors(LabelSelectorConverter.selectorsToNewProto(request.getSelectors()))
            .setTextSearch(request.getTextSearch())
            .setValidationFilter(request.getValidationFilter())
            .setLimit(request.getLimit())
            .setDeadlineMillis(request.getDeadline())
            .build();
    }

    public static TLabelNamesRequest toProto(LabelNamesRequest request) {
        TLabelNamesRequest.Builder builder = TLabelNamesRequest.newBuilder()
            .setDeadlineMillis(request.getDeadline());

        boolean useNewFormat = request.isUseNewFormat() || !request.getSelectors().getNameSelector().isEmpty();

        if (useNewFormat) {
            builder.setNewSelectors(LabelSelectorConverter.selectorsToNewProto(request.getSelectors()));
        } else {
            builder.addAllSelectors(LabelSelectorConverter.selectorsToProto(request.getSelectors()));
        }

        builder.setFolderId(request.getFolderId());

        return builder.build();
    }

    public static TLabelValuesRequest toProto(LabelValuesRequest request) {
        TLabelValuesRequest.Builder builder = TLabelValuesRequest.newBuilder()
            .setTextSearch(request.getTextSearch())
            .setValidationFilter(request.getValidationFilter())
            .setLimit(request.getLimit())
            .addAllLabels(request.getLabelNames())
            .setFolderId(request.getFolderId())
            .setDeadlineMillis(request.getDeadline());

        boolean useNewFormat = request.isUseNewFormat() || !request.getSelectors().getNameSelector().isEmpty();

        if (useNewFormat) {
            builder.setNewSelectors(LabelSelectorConverter.selectorsToNewProto(request.getSelectors()));
        } else {
            builder.addAllSelectors(LabelSelectorConverter.selectorsToProto(request.getSelectors()));
        }

        return builder.build();
    }

    public static TUniqueLabelsRequest toProto(UniqueLabelsRequest request) {
        TUniqueLabelsRequest.Builder builder = TUniqueLabelsRequest.newBuilder()
            .setDeadlineMillis(request.getDeadline())
            .addAllNames(request.getLabels());

        boolean useNewFormat = request.isUseNewFormat() || !request.getSelectors().getNameSelector().isEmpty();

        if (useNewFormat) {
            builder.setNewSelectors(LabelSelectorConverter.selectorsToNewProto(request.getSelectors()));
        } else {
            builder.addAllSelectors(LabelSelectorConverter.selectorsToProto(request.getSelectors()));
        }

        return builder.build();
    }

    public static ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest toProto(ResolveOneRequest request) {
        return ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest.newBuilder()
            .addAllLabels(LabelConverter.labelsToProtoList(request.getLabels()))
            .setDeadlineMillis(request.getDeadline())
            .build();
    }

    public static ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest toProto(ResolveOneWithNameRequest request) {
        return ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest.newBuilder()
            .setName(request.getMetric().getName())
            .addAllLabels(LabelConverter.labelsToProtoList(request.getMetric().getLabels()))
            .setDeadlineMillis(request.getDeadline())
            .build();
    }

    public static ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest toProto(ResolveManyRequest request) {
        List<Labels> protoLabelsList = request.getLabelsList().stream()
                .map(LabelConverter::labelsToProto)
                .collect(toList());

        List<Label> protoCommonLabels =
            LabelConverter.labelsToProtoList(request.getCommonLabels());

        return ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest.newBuilder()
                .addAllCommonLabels(protoCommonLabels)
                .addAllListLabels(protoLabelsList)
                .setDeadlineMillis(request.getDeadline())
                .build();
    }

    public static ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest toProto(ResolveManyWithNameRequest request) {
        List<MetabaseMetricId> protoMetabaseMetricIds = request.getMetrics().stream()
                .map(metric -> MetabaseMetricId.newBuilder()
                    .setName(metric.getName())
                    .addAllLabels(LabelConverter.labelsToProtoList(metric.getLabels()))
                    .build())
                .collect(toList());

        List<Label> protoCommonLabels =
            LabelConverter.labelsToProtoList(request.getCommonLabels());

        return ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest.newBuilder()
                .addAllCommonLabels(protoCommonLabels)
                .addAllMetrics(protoMetabaseMetricIds)
                .setDeadlineMillis(request.getDeadline())
                .build();
    }

    public static TReadRequest toProto(ReadRequest request, StockpileKey key, StockpileFormat format) {
        return TReadRequest.newBuilder()
                .setMetricId(toProto(key))
                .setDeadline(request.getDeadline())
                .setFromMillis(request.getFromMillis())
                .setToMillis(request.getToMillis())
                .setGridMillis(request.getGridMillis())
                .setAggregation(request.getAggregation())
                .setBinaryVersion(format.getFormat())
                .setProducer(request.getProducer())
                .build();
    }

    public static MetricId toProto(StockpileKey key) {
        return MetricId.newBuilder()
                .setShardId(key.getShardId())
                .setLocalId(key.getLocalId())
                .build();
    }
}
