package ru.yandex.solomon.metrics.client;

import java.time.Clock;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.WillCloseWhenClosed;

import com.google.common.annotations.VisibleForTesting;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metrics.client.combined.CombinedCall;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyResponse;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethodArgument;
import ru.yandex.solomon.staffOnly.manager.special.DurationMillis;
import ru.yandex.solomon.util.client.ClientFutures;
import ru.yandex.solomon.util.client.StatusAware;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class CrossDcMetricsClient implements MetricsClient {

    private final Clock clock;
    @WillCloseWhenClosed
    private final Map<String, MetricsClient> clientByDestination;
    @DurationMillis
    private volatile long slowResponseAwaitMillis = 3_000;
    @DurationMillis
    private volatile long slowResponseAwaitMaxLimitMillis = 60_000;

    @VisibleForTesting
    public CrossDcMetricsClient(Clock clock, @WillCloseWhenClosed Map<String, MetricsClient> clientByDestination) {
        this.clock = clock;
        this.clientByDestination = clientByDestination;
    }

    public CrossDcMetricsClient(@WillCloseWhenClosed Map<String, MetricsClient> clientByDestination) {
        this(Clock.systemUTC(), clientByDestination);
    }

    @ManagerMethod
    void setSlowResponseAwaitMillis(@ManagerMethodArgument(name = "timeMillis") long timeMillis) {
        this.slowResponseAwaitMillis = timeMillis;
    }

    @ManagerMethod
    void setSlowResponseAwaitMaxLimitMillis(@ManagerMethodArgument(name = "timeMillis") long timeMillis) {
        this.slowResponseAwaitMaxLimitMillis = timeMillis;
    }

    private <T extends StatusAware> Function<List<CompletableFuture<T>>, CompletableFuture<List<T>>> waitSome(
            AbstractRequest request)
    {
        return futures -> allOrAnyOkOf(futures, request.getSoftDeadline());
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        if (request.getOffset() > 0 && StringUtils.isEmpty(request.getDestination())) {
            MetabaseStatus status = MetabaseStatus.fromCode(
                EMetabaseStatusCode.INVALID_REQUEST,
                "pagination isn't supported for cross-destination requests");
            return completedFuture(new FindResponse(status));
        }

        return resolveClient(request)
                .map(client -> client.find(request))
                .collect(collectingAndThen(toList(), waitSome(request)))
                .thenApply(responses -> CrossDcResponseMerger.mergeFindResponses(request, responses, totalDestinationsCount(request)));
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        return resolveClient(request)
            .map(client -> client.resolveOne(request))
            .collect(collectingAndThen(toList(), waitSome(request)))
            .thenApply(responses -> CrossDcResponseMerger.processResolveOneResponse(request, responses));
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOneWithName(
        ResolveOneWithNameRequest request)
    {
        return resolveClient(request)
            .map(client -> client.resolveOneWithName(request))
            .collect(collectingAndThen(toList(), waitSome(request)))
            .thenApply(responses -> CrossDcResponseMerger.processResolveOneResponse(request, responses));
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        return resolveClient(request)
            .map(client -> client.resolveMany(request))
            .collect(collectingAndThen(toList(), waitSome(request)))
            .thenApply(responses -> CrossDcResponseMerger.processResolveManyResponse(request, responses));
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveManyWithName(
        ResolveManyWithNameRequest request)
    {
        return resolveClient(request)
            .map(client -> client.resolveManyWithName(request))
            .collect(collectingAndThen(toList(), waitSome(request)))
            .thenApply(responses -> CrossDcResponseMerger.processResolveManyResponse(request, responses));
    }

    @Override
    public CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request) {
        return resolveClient(request)
            .map(client -> client.metricNames(request))
            .collect(collectingAndThen(toList(), waitSome(request)))
            .thenApply(responses -> CrossDcResponseMerger.mergeMetricNamesResponses(request, responses));
    }

    @Override
    public CompletableFuture<LabelNamesResponse> labelNames(LabelNamesRequest request) {
        return resolveClient(request)
                .map(client -> client.labelNames(request))
                .collect(collectingAndThen(toList(), waitSome(request)))
                .thenApply(responses -> CrossDcResponseMerger.mergeLabelNamesResponses(request, responses));
    }

    @Override
    public CompletableFuture<LabelValuesResponse> labelValues(LabelValuesRequest request) {
        return resolveClient(request)
                .map(client -> client.labelValues(request))
                .collect(collectingAndThen(toList(), waitSome(request)))
                .thenApply(responses -> CrossDcResponseMerger.mergeLabelValuesResponses(request, responses));
    }

    @Override
    public CompletableFuture<UniqueLabelsResponse> uniqueLabels(UniqueLabelsRequest request) {
        var dest = request.getDestination();
        return resolveClient(request)
                .map(client -> client.uniqueLabels(request))
                .collect(collectingAndThen(toList(), waitSome(request)))
                .thenApply(responses -> CrossDcResponseMerger.mergeUniqueLabelsResponses(dest, responses));
    }

    @Override
    public CompletableFuture<ReadResponse> read(ReadRequest request) {
        return resolveClient(request)
                .map(client -> client.read(request))
                .collect(collectingAndThen(toList(), waitSome(request)))
                .thenApply(responses -> CrossDcResponseMerger.mergeReadResponses(request, responses, totalDestinationsCount(request)));
    }

    @Override
    public CompletableFuture<ReadManyResponse> readMany(ReadManyRequest request) {
        return resolveClient(request)
                .map(client -> client.readMany(request))
                .collect(collectingAndThen(toList(), waitSome(request)))
                .thenApply(responses -> CrossDcResponseMerger.mergeReadManyResponses(request, responses));
    }

    @Override
    public CompletableFuture<FindAndReadManyResponse> findAndReadMany(FindAndReadManyRequest request) {
        return CombinedCall.defaultFindAndReadMany(this, request);
    }

    @Override
    public AvailabilityStatus getMetabaseAvailability() {
        return clientByDestination.values()
                .stream()
                .map(MetricsClient::getMetabaseAvailability)
                .max(Comparator.comparingDouble(AvailabilityStatus::getAvailability))
                .orElse(AvailabilityStatus.UNAVAILABLE);
    }

    @Override
    public AvailabilityStatus getStockpileAvailability() {
        return clientByDestination.values()
                .stream()
                .map(MetricsClient::getStockpileAvailability)
                .max(Comparator.comparingDouble(AvailabilityStatus::getAvailability))
                .orElse(AvailabilityStatus.UNAVAILABLE);
    }

    @Override
    public Stream<Labels> metabaseShards(String destination, Selectors selector) {
        return resolveClient(destination)
                .flatMap(client -> client.metabaseShards(destination, selector))
                .distinct();
    }

    @Override
    public String getStockpileHostForShardId(String destination, int shardId) {
        MetricsClient client = clientByDestination.get(destination);

        if (client == null) {
            throw new IllegalArgumentException("Not found destination: " + destination);
        }

        return client.getStockpileHostForShardId(destination, shardId);
    }

    @Override
    public void close() {
        clientByDestination.values().forEach(MetricsClient::close);
    }

    private <T extends StatusAware> CompletableFuture<List<T>> allOrAnyOkOf(
            List<CompletableFuture<T>> futures,
            long softDeadline)
    {
        return ClientFutures.allOrAnyOkOf(
            futures,
            () -> softDeadline == 0
                ? slowResponseAwaitMillis
                : Math.min(softDeadline - clock.millis(), slowResponseAwaitMaxLimitMillis));
    }

    @Override
    public Collection<String> getDestinations() {
        return clientByDestination.keySet();
    }

    private int totalDestinationsCount(AbstractRequest request) {
        if (request.getDestination() == null) {
            return clientByDestination.size();
        }
        return clientByDestination.containsKey(request.getDestination()) ? 1 : 0;
    }

    private Stream<MetricsClient> resolveClient(Set<String> destinations) {
        if (destinations.isEmpty() || destinations.equals(clientByDestination.keySet())) {
            return clientByDestination.values().stream();
        }

        return clientByDestination.entrySet()
                .stream()
                .filter(entry -> destinations.contains(entry.getKey()))
                .map(Map.Entry::getValue);
    }

    private Stream<MetricsClient> resolveClient(AbstractRequest request) {
        return resolveClient(request.getDestinations());
    }

    private Stream<MetricsClient> resolveClient(String destination) {
        return resolveClient(destination == null ? Set.of() : Set.of(destination));
    }
}
