package ru.yandex.solomon.metrics.client;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.MergingAggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.aggregation.TimeseriesSummary;
import ru.yandex.solomon.util.labelStats.LabelStats;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;
import ru.yandex.stockpile.api.EStockpileStatusCode;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class CrossDcResponseMerger {

    private CrossDcResponseMerger() {
    }

    public static FindResponse mergeFindResponses(FindRequest request, List<FindResponse> responses, int totalDestinationCount) {
        ensureDestinationResponses(responses, request);
        if (responses.size() == 1) {
            var one = responses.get(0);
            return new FindResponse(
                one.getStatus(),
                one.getMetrics(),
                one.getTotalCount(),
                one.getMetricsCountByDestination(),
                one.isTruncated(),
                one.isOk() && totalDestinationCount == 1);
        }

        MetabaseStatus status = ensureAtLeastOneMetaSuccess(responses);
        if (status != null) {
            return new FindResponse(status);
        }

        Map<Pair<String, Labels>, MetricKey> result = new HashMap<>();
        int totalCount = 0;
        Map<String, Integer> metricsCountByDestination = new HashMap<>(responses.size());

        boolean truncated = false;
        int cntSuccess = 0;
        for (FindResponse response : responses) {
            if (!response.isOk()) {
                continue;
            }

            cntSuccess++;
            truncated |= response.isTruncated();
            for (MetricKey key : response.getMetrics()) {
                Pair<String, Labels> nameAndLabels = Pair.of(key.getName(), key.getLabels());
                MetricKey prev = result.get(nameAndLabels);
                if (prev == null) {
                    if (result.size() >= request.getLimit()) {
                        truncated = true;
                        continue;
                    }

                    result.put(nameAndLabels, key);
                } else {
                    result.put(nameAndLabels, prev.combine(key));
                }
            }

            totalCount += response.getTotalCount();

            metricsCountByDestination.putAll(response.getMetricsCountByDestination());
        }

        return new FindResponse(
            List.copyOf(result.values()),
            totalCount,
            metricsCountByDestination,
            truncated,
            cntSuccess == totalDestinationCount
        );
    }

    public static ResolveOneResponse processResolveOneResponse(AbstractRequest request, List<ResolveOneResponse> responses) {
        ensureDestinationResponses(responses, request);
        if (responses.size() == 1) {
            return responses.get(0);
        }

        MetabaseStatus status = ensureAtLeastOneMetaSuccess(responses);
        if (status != null) {
            return new ResolveOneResponse(status);
        }

        MetricKey result = null;
        for (ResolveOneResponse response : responses) {
            if (!response.isOk()) {
                continue;
            }

            if (result == null) {
                result = response.getMetric();
            } else {
                result = result.combine(response.getMetric());
            }
        }

        if (result == null) {
            final String description;

            if (request instanceof ResolveOneWithNameRequest) {
                description = "Failed to find metric: " + ((ResolveOneWithNameRequest) request).getMetric();
            } else if (request instanceof ResolveOneRequest) {
                description = "Failed to find labels: " + ((ResolveOneRequest) request).getLabels();
            } else {
                description = "";
            }

            MetabaseStatus notFoundStatus = MetabaseStatus.fromCode(
                EMetabaseStatusCode.NOT_FOUND,
                description);

            return new ResolveOneResponse(notFoundStatus);
        }

        return new ResolveOneResponse(result);
    }

    public static ResolveManyResponse processResolveManyResponse(AbstractRequest request, List<ResolveManyResponse> responses) {
        ensureDestinationResponses(responses, request);
        if (responses.size() == 1) {
            return responses.get(0);
        }

        MetabaseStatus status = ensureAtLeastOneMetaSuccess(responses);
        if (status != null) {
            return new ResolveManyResponse(status);
        }

        Map<Pair<String, Labels>, MetricKey> result = new HashMap<>();
        for (ResolveManyResponse response : responses) {
            if (!response.isOk()) {
                continue;
            }

            for (MetricKey key : response.getMetrics()) {
                Pair<String, Labels> nameAndLabels = Pair.of(key.getName(), key.getLabels());
                MetricKey prev = result.get(nameAndLabels);
                if (prev == null) {
                    result.put(nameAndLabels, key);
                } else {
                    result.put(nameAndLabels, prev.combine(key));
                }
            }
        }

        return new ResolveManyResponse(List.copyOf(result.values()));
    }

    public static MetricNamesResponse mergeMetricNamesResponses(MetricNamesRequest request, List<MetricNamesResponse> responses) {
        ensureDestinationResponses(responses, request);
        if (responses.size() == 1) {
            return responses.get(0);
        }

        MetabaseStatus status = ensureAtLeastOneMetaSuccess(responses);
        if (status != null) {
            return new MetricNamesResponse(status);
        }

        LabelStats result = LabelStats.create();
        for (MetricNamesResponse response : responses) {
            if (!response.isOk()) {
                continue;
            }

            LabelStats stats = LabelStats.create();
            stats.addAll(response.getNames(), 0, response.isTruncated());
            result.combine(stats);
        }

        result.limit(request.getLimit());
        return new MetricNamesResponse(List.copyOf(result.getValues()), result.isTruncated());
    }

    public static LabelNamesResponse mergeLabelNamesResponses(LabelNamesRequest request, List<LabelNamesResponse> responses) {
        ensureDestinationResponses(responses, request);
        if (responses.size() == 1) {
            return responses.get(0);
        }

        MetabaseStatus status = ensureAtLeastOneMetaSuccess(responses);
        if (status != null) {
            return new LabelNamesResponse(status);
        }

        Set<String> result = new HashSet<>();
        for (LabelNamesResponse response : responses) {
            if (!response.isOk()) {
                continue;
            }

            result.addAll(response.getNames());
        }

        return new LabelNamesResponse(result);
    }

    public static LabelValuesResponse mergeLabelValuesResponses(LabelValuesRequest request, List<LabelValuesResponse> responses) {
        ensureDestinationResponses(responses, request);
        if (responses.size() == 1) {
            return responses.get(0);
        }

        MetabaseStatus status = ensureAtLeastOneMetaSuccess(responses);
        if (status != null) {
            return new LabelValuesResponse(status);
        }

        LabelValuesStats result = new LabelValuesStats();
        Map<String, Integer> metricsCountByDestination = new HashMap<>(responses.size());
        for (LabelValuesResponse response : responses) {
            if (!response.isOk()) {
                continue;
            }

            result.combine(response.getLabelValuesStats());

            metricsCountByDestination.putAll(response.getMetricsCountByDestination());
        }

        result.limit(request.getLimit());
        return new LabelValuesResponse(result, metricsCountByDestination);
    }

    public static UniqueLabelsResponse mergeUniqueLabelsResponses(String dest, List<UniqueLabelsResponse> responses) {
        ensureDestinationResponses(responses, dest);
        if (responses.size() == 1) {
            return responses.get(0);
        }

        MetabaseStatus status = ensureAtLeastOneMetaSuccess(responses);
        if (status != null) {
            return new UniqueLabelsResponse(status);
        }

        Set<Labels> merged = new HashSet<>();
        boolean allDestSuccess = true;
        for (UniqueLabelsResponse response : responses) {
            if (!response.isOk()) {
                allDestSuccess = false;
                continue;
            }

            merged.addAll(response.getUniqueLabels());
        }

        return new UniqueLabelsResponse(MetabaseStatus.OK, merged, allDestSuccess);
    }

    public static ReadResponse mergeReadResponses(ReadRequest request, List<ReadResponse> responses, int totalDestinationsCount) {
        ensureDestinationResponses(responses, request);
        if (responses.size() == 1) {
            ReadResponse one = responses.get(0);
            return new ReadResponse(
                one.getMetric(),
                one.getStatus(),
                one.isAllDestSuccess() && totalDestinationsCount == 1
            );
        }

        StockpileStatus status = ensureAtLeastOneDataSuccess(responses);
        if (status != null) {
            return new ReadResponse(request.getKey(), status);
        }

        var sourcesFromOkResponses = responses.stream()
                .filter(StockpileStatusAware::isOk)
                .map(ReadResponse::getMetric)
                .collect(toList());

        Metric<MetricKey> merged = mergeMetrics(sourcesFromOkResponses);
        boolean allDestSuccess = sourcesFromOkResponses.size() == totalDestinationsCount;

        return new ReadResponse(merged, StockpileStatus.OK, allDestSuccess);
    }

    public static ReadManyResponse mergeReadManyResponses(ReadManyRequest request, List<ReadManyResponse> responses) {
        ensureDestinationResponses(responses, request);
        if (responses.size() == 1) {
            return responses.get(0);
        }

        StockpileStatus status = ensureAtLeastOneDataSuccess(responses);
        if (status != null) {
            return new ReadManyResponse(status);
        }

        Map<MetricKey, Metric<MetricKey>> result = responses.stream()
                .filter(StockpileStatusAware::isOk)
                .flatMap(readManyResponse -> readManyResponse.getMetrics().stream())
                .collect(groupingBy(s -> MetricKey.orUnknown(s.getKey()), collectingAndThen(toList(), CrossDcResponseMerger::mergeMetrics)));

        return new ReadManyResponse(List.copyOf(result.values()));
    }

    private static Metric<MetricKey> mergeMetrics(List<Metric<MetricKey>> metrics) {
        if (metrics.size() == 1) {
            return metrics.get(0);
        }

        @Nullable
        MetricKey key = metrics.get(0).getKey();
        AggrGraphDataIterable mergedTimeseries = metrics.stream()
                .map(Metric::getTimeseries)
                .filter(Objects::nonNull)
                .collect(collectingAndThen(toList(), MergingAggrGraphDataIterable::of));

        @Nullable
        TimeseriesSummary summary = metrics.stream()
                .map(Metric::getSummary)
                .filter(Objects::nonNull)
                .max((o1, o2) -> Long.compareUnsigned(o1.getCount(), o2.getCount()))
                .orElse(null);

        MetricType type = metrics.stream()
                .map(Metric::getType)
                .filter(k -> k != MetricType.METRIC_TYPE_UNSPECIFIED)
                .findFirst()
                .orElse(MetricType.METRIC_TYPE_UNSPECIFIED);

        return new Metric<>(key, type, mergedTimeseries, summary);
    }

    @Nullable
    private static MetabaseStatus ensureAtLeastOneMetaSuccess(List<? extends MetabaseStatusAware> responses) {
        EMetabaseStatusCode commonStatus = EMetabaseStatusCode.UNKNOWN;
        for (MetabaseStatusAware response : responses) {
            if (response.isOk()) {
                return null;
            }

            if (commonStatus != EMetabaseStatusCode.NOT_FOUND) {
                commonStatus = response.getStatus().getCode();
            }
        }

        return MetabaseStatus.fromCode(commonStatus, "Response not OK from all clusters\n" + responses.stream()
                .map(response -> response.getStatus().toString())
                .collect(Collectors.joining("\n")));
    }

    @Nullable
    private static StockpileStatus ensureAtLeastOneDataSuccess(List<? extends StockpileStatusAware> responses) {
        EStockpileStatusCode commonStatus = EStockpileStatusCode.UNKNOWN;
        for (StockpileStatusAware response : responses) {
            if (response.isOk()) {
                return null;
            }

            if (commonStatus != EStockpileStatusCode.METRIC_NOT_FOUND) {
                commonStatus = response.getStatus().getCode();
            }
        }

        return StockpileStatus.fromCode(commonStatus, "Response not OK from all clusters\n" + responses.stream()
                .map(response -> response.getStatus().toString())
                .collect(Collectors.joining("\n")));
    }

    private static <T> void ensureDestinationResponses(List<T> responses, AbstractRequest request) {
        if (responses.isEmpty()) {
            throw new IllegalArgumentException("Not found destination: " + request.getDestination());
        }
    }

    private static <T> void ensureDestinationResponses(List<T> responses, String dest) {
        if (responses.isEmpty()) {
            throw new IllegalArgumentException("Not found destination: " + dest);
        }
    }
}
