package ru.yandex.solomon.metrics.client;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.WillCloseWhenClosed;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metrics.client.combined.CombinedCall;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyResponse;
import ru.yandex.solomon.metrics.client.dc.DcCallReadMany;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.StockpileKey;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.solomon.util.labelStats.LabelStatsConverter;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.client.StockpileClient;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.metrics.client.Converters.toProto;

/**
 * @author Vladimir Gordiychuk
 */
public class DcMetricsClient implements MetricsClient {
    private static final Logger logger = LoggerFactory.getLogger(DcMetricsClient.class);
    private final String destination;
    @WillCloseWhenClosed
    private final MetabaseClient metabase;
    @WillCloseWhenClosed
    private final StockpileClient stockpile;
    private final LabelAllocator labelAllocator;

    public DcMetricsClient(
            String destination,
            @WillCloseWhenClosed MetabaseClient metabase,
            @WillCloseWhenClosed StockpileClient stockpile)
    {
        this(destination, metabase, stockpile, Labels.allocator);
    }

    public DcMetricsClient(
        String destination,
        @WillCloseWhenClosed MetabaseClient metabase,
        @WillCloseWhenClosed StockpileClient stockpile,
        LabelAllocator labelAllocator)
    {
        this.destination = destination;
        this.metabase = metabase;
        this.stockpile = stockpile;
        this.labelAllocator = labelAllocator;
    }

    public String getDestination() {
        return destination;
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        return metabase.find(toProto(request))
                .thenApply(response -> {
                    if (response.getStatus() != EMetabaseStatusCode.OK) {
                        MetabaseStatus status = MetabaseStatus.fromCode(response.getStatus(), "[" + destination + "] " + response.getStatusMessage());

                        if (logger.isDebugEnabled()) {
                            logger.debug("Metabase error({}) - {} for selectors {}", destination, status, Selectors.format(request.getSelectors()));
                        }
                        return new FindResponse(status);
                    }

                    List<MetricKey> metrics = protoToKeys(response.getMetricsList());

                    boolean truncated = response.getMetricsCount() < response.getTotalCount();

                    Map<String, Integer> countByDestination =
                        Collections.singletonMap(destination, response.getTotalCount());

                    return new FindResponse(MetabaseStatus.OK, metrics, response.getTotalCount(), countByDestination, truncated, true);
                })
                .exceptionally(e -> {
                    MetabaseStatus status = MetabaseStatus.fromThrowable(e);
                    logger.error("Uncaught metabase error({}) - {} for selectors {}", destination, status, request.getSelectors());
                    return new FindResponse(status);
                });
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        return metabase.resolveOne(toProto(request))
            .thenApply(response -> {
                if (response.getStatus() != EMetabaseStatusCode.OK) {
                    MetabaseStatus status = MetabaseStatus.fromCode(response.getStatus(), "[" + destination + "] " + response.getStatusMessage());
                    if (logger.isDebugEnabled()) {
                        logger.debug("Metabase error({}) - {} for labels {}", destination, status, request.getLabels());
                    }
                    return new ResolveOneResponse(status);
                }

                MetricKey metric = protoToKey(response.getMetric());
                return new ResolveOneResponse(metric);
            })
            .exceptionally(e -> {
                MetabaseStatus status = MetabaseStatus.fromThrowable(e);
                logger.error("Uncaught metabase error({}) - {} for labels {}", destination, status, request.getLabels());
                return new ResolveOneResponse(status);
            });
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOneWithName(
        ResolveOneWithNameRequest request)
    {
        return metabase.resolveOne(toProto(request))
            .thenApply(response -> {
                if (response.getStatus() != EMetabaseStatusCode.OK) {
                    MetabaseStatus status = MetabaseStatus.fromCode(response.getStatus(), "[" + destination + "] " + response.getStatusMessage());
                    if (logger.isDebugEnabled()) {
                        logger.debug("Metabase error({}) - {} for metrics {}", destination, status, request.getMetric());
                    }
                    return new ResolveOneResponse(status);
                }

                MetricKey metric = protoToKey(response.getMetric());
                return new ResolveOneResponse(metric);
            })
            .exceptionally(e -> {
                MetabaseStatus status = MetabaseStatus.fromThrowable(e);
                logger.error("Uncaught metabase error({}) - {} for metrics {}", destination, status, request.getMetric());
                return new ResolveOneResponse(status);
            });
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        return metabase.resolveMany(toProto(request))
            .thenApply(response -> {
                if (response.getStatus() != EMetabaseStatusCode.OK) {
                    MetabaseStatus status = MetabaseStatus.fromCode(response.getStatus(), "[" + destination + "] " + response.getStatusMessage());
                    if (logger.isDebugEnabled()) {
                        logger.debug("Metabase error({}) - {} for labels {}", destination, status, request.getLabelsList());
                    }
                    return new ResolveManyResponse(status);
                }

                List<MetricKey> metrics = response.getMetricsList()
                    .stream()
                    .map(this::protoToKey)
                    .collect(toList());

                return new ResolveManyResponse(metrics);
            })
            .exceptionally(e -> {
                MetabaseStatus status = MetabaseStatus.fromThrowable(e);
                logger.error("Uncaught metabase error({}) - {} for labels {}", destination, status, request.getLabelsList());
                return new ResolveManyResponse(status);
            });
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveManyWithName(
        ResolveManyWithNameRequest request)
    {
        return metabase.resolveMany(toProto(request))
            .thenApply(response -> {
                if (response.getStatus() != EMetabaseStatusCode.OK) {
                    MetabaseStatus status = MetabaseStatus.fromCode(response.getStatus(), "[" + destination + "] " + response.getStatusMessage());
                    if (logger.isDebugEnabled()) {
                        logger.debug("Metabase error({}) - {} for metrics {}", destination, status, request.getMetrics());
                    }
                    return new ResolveManyResponse(status);
                }

                List<MetricKey> metrics = response.getMetricsList()
                    .stream()
                    .map(this::protoToKey)
                    .collect(toList());

                return new ResolveManyResponse(metrics);
            })
            .exceptionally(e -> {
                MetabaseStatus status = MetabaseStatus.fromThrowable(e);
                logger.error("Uncaught metabase error({}) - {} for metrics {}", destination, status, request.getMetrics());
                return new ResolveManyResponse(status);
            });
    }

    @Override
    public CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request) {
        return metabase.metricNames(toProto(request))
            .thenApply(response -> {
                if (response.getStatus() != EMetabaseStatusCode.OK) {
                    MetabaseStatus status = MetabaseStatus.fromCode(response.getStatus(), "[" + destination + "] " + response.getStatusMessage());
                    if (logger.isDebugEnabled()) {
                        logger.debug("Metabase error({}) - metricNames {} for request {}", destination, status, request);
                    }

                    return new MetricNamesResponse(status);
                }

                return new MetricNamesResponse(response.getNamesList(), response.getTruncated());
            });
    }

    @Override
    public CompletableFuture<LabelNamesResponse> labelNames(LabelNamesRequest request) {
        return metabase.labelNames(toProto(request))
                .thenApply(response -> {
                    if (response.getStatus() != EMetabaseStatusCode.OK) {
                        MetabaseStatus status = MetabaseStatus.fromCode(response.getStatus(), "[" + destination + "] " + response.getStatusMessage());
                        if (logger.isDebugEnabled()) {
                            logger.debug("Metabase error({}) - labelNames {} for selectors {}", destination, status, Selectors.format(request.getSelectors()));
                        }

                        return new LabelNamesResponse(status);
                    }

                    return new LabelNamesResponse(new HashSet<>(response.getNamesList()));
                });
    }

    @Override
    public CompletableFuture<LabelValuesResponse> labelValues(LabelValuesRequest request) {
        return metabase.labelValues(toProto(request))
                .thenApply(response -> {
                    if (response.getStatus() != EMetabaseStatusCode.OK) {
                        MetabaseStatus status = MetabaseStatus.fromCode(response.getStatus(), "[" + destination + "] " + response.getStatusMessage());
                        if (logger.isDebugEnabled()) {
                            logger.debug("Metabase error({}) - labelValues {} for request {}", destination, status, request);
                        }

                        return new LabelValuesResponse(status);
                    }

                    Map<String, Integer> metricsCountByDestination =
                        Collections.singletonMap(destination, response.getMetricCount());

                    return new LabelValuesResponse(LabelStatsConverter.fromProto(response), metricsCountByDestination);
                });
    }

    @Override
    public CompletableFuture<UniqueLabelsResponse> uniqueLabels(UniqueLabelsRequest request) {
        return metabase.uniqueLabels(toProto(request))
                .thenApply(response -> {
                    if (response.getStatus() != EMetabaseStatusCode.OK) {
                        MetabaseStatus status = MetabaseStatus.fromCode(response.getStatus(), "[" + destination + "] " + response.getStatusMessage());
                        if (logger.isDebugEnabled()) {
                            logger.debug("Metabase error({}) on resolve unique labels {}", destination, status);
                        }
                        return new UniqueLabelsResponse(status);
                    }

                    return new UniqueLabelsResponse(response.getLabelListsList()
                            .stream()
                            .map(proto -> LabelConverter.protoToLabels(proto.getLabelsList(), labelAllocator))
                            .collect(Collectors.toSet()));
                });
    }

    @Override
    public CompletableFuture<ReadResponse> read(ReadRequest request) {
        MetricKey key = request.getKey();
        StockpileKey stockpileKey = getStockpileKey(key);
        if (stockpileKey == null) {
            StockpileStatus status = StockpileStatus.fromCode(EStockpileStatusCode.METRIC_NOT_FOUND,
                    "Sensor " + key + " absent at " + destination);
            return completedFuture(new ReadResponse(key, status));
        }

        StockpileFormat format = defineStockpileFormat();
        return stockpile.readCompressedOne(toProto(request, stockpileKey, format))
                .thenApply(response -> {
                    if (response.getStatus() != EStockpileStatusCode.OK) {
                        StockpileStatus status = StockpileStatus.fromCode(response.getStatus(), "[" + destination + "] " + response.getStatusMessage());
                        if (logger.isDebugEnabled()) {
                            logger.debug("Stockpile error({}) - {} for key {}", destination, status, key);
                        }

                        return new ReadResponse(key, status);
                    }

                    AggrGraphDataIterable timeseries = TimeSeriesCodec.sequenceDecode(response);

                    MetricType type = response.getType();
                    if (type == MetricType.METRIC_TYPE_UNSPECIFIED) {
                        type = StockpileColumns.typeByMask(timeseries.columnSetMask());
                    }

                    var metric = new Metric<>(key, type, timeseries);
                    return new ReadResponse(metric, StockpileStatus.OK, true);
                })
                .exceptionally(e -> {
                    StockpileStatus status = StockpileStatus.fromThrowable(e);
                    logger.error("Uncaught stockpile error({}) - {} for key {}", destination, status, key);
                    return new ReadResponse(key, status);
                });
    }

    @Override
    public CompletableFuture<ReadManyResponse> readMany(ReadManyRequest request) {
        return DcCallReadMany.of(destination, stockpile, request).call();
    }

    @Override
    public AvailabilityStatus getMetabaseAvailability() {
        return metabase.getAvailability();
    }

    @Override
    public AvailabilityStatus getStockpileAvailability() {
        return stockpile.getAvailability();
    }

    @Override
    public Stream<Labels> metabaseShards(String destination, Selectors selectors) {
        return metabase.shards(selectors);
    }

    @Override
    public CompletableFuture<FindAndReadManyResponse> findAndReadMany(FindAndReadManyRequest request) {
        return CombinedCall.defaultFindAndReadMany(this, request);
    }

    @Override
    public String getStockpileHostForShardId(String destination, int shardId) {
        return stockpile.getHostForShardId(shardId);
    }

    @Override
    public Collection<String> getDestinations() {
        return List.of(destination);
    }

    @Override
    public void close() {
        metabase.close();
        stockpile.close();
    }

    @Nullable
    private StockpileKey getStockpileKey(MetricKey metricKey) {
        for (StockpileKey key : metricKey.getStockpileKeys()) {
            if (destination.equals(key.getDestination())) {
                return key;
            }
        }

        return null;
    }

    private StockpileFormat defineStockpileFormat() {
        return StockpileFormat.byNumberOrCurrent(stockpile.getCompatibleCompressFormat().upperEndpoint());
    }

    private List<MetricKey> protoToKeys(List<ru.yandex.solomon.metabase.api.protobuf.Metric> list) {
        switch (list.size()) {
            case 0:
                return List.of();
            case 1:
                return List.of(protoToKey(list.get(0)));
            default:
                var result = new ArrayList<MetricKey>(list.size());
                for (var proto : list) {
                    result.add(protoToKey(proto));
                }
                return result;
        }
    }

    private MetricKey protoToKey(ru.yandex.solomon.metabase.api.protobuf.Metric proto) {
        var type = MetricTypeConverter.fromProto(proto.getType());
        String name = internName(proto.getName());
        Labels labels = LabelConverter.protoToLabels(proto.getLabelsList(), labelAllocator);
        StockpileKey stockpileKey = protoToKey(proto.getMetricId());
        long createdAtMillis = proto.getCreatedAtMillis();
        return new MetricKey(type, name, labels, createdAtMillis, List.of(stockpileKey));
    }

    private StockpileKey protoToKey(MetricId proto) {
        return new StockpileKey(destination, proto.getShardId(), proto.getLocalId());
    }

    private String internName(String name) {
        if (labelAllocator instanceof InterningLabelAllocator l) {
            return l.interValue(name);
        }

        return name;
    }
}
