package ru.yandex.solomon.metrics.client;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nullable;

import com.google.common.base.Ticker;

import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.metabase.client.TMetabaseClientConfig;
import ru.yandex.solomon.dataproxy.client.DataProxyClient;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.metrics.client.cache.CachingMetricsClientImpl;
import ru.yandex.solomon.metrics.client.cache.FindCacheOptions;
import ru.yandex.solomon.metrics.client.cache.MetabaseFindCache;
import ru.yandex.solomon.metrics.client.cache.MetabaseFindCacheImpl;
import ru.yandex.solomon.metrics.client.cache.UnrollClient;
import ru.yandex.solomon.metrics.client.cache.UnrollClientImpl;
import ru.yandex.solomon.metrics.client.cache.UnrollClientOptions;
import ru.yandex.solomon.metrics.client.dataproxy.DataProxyAwareClient;
import ru.yandex.stockpile.client.StockpileClient;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricsClients {
    public static MetricsClient create(String dest, MetabaseClient metabase, StockpileClient stockpile) {
        return create(dest, metabase, stockpile, Labels.allocator);
    }

    public static MetricsClient create(String dest, MetabaseClient metabase, StockpileClient stockpile, LabelAllocator labelAllocator) {
        return new DcMetricsClient(dest, metabase, stockpile, labelAllocator);
    }

    public static MetricsClient create(Map<String, MetabaseClient> metabase, Map<String, StockpileClient> stockpile) {
        return create(metabase, stockpile, Labels.allocator);
    }

    public static MetricsClient create(
            Map<String, MetabaseClient> metabase,
            Map<String, StockpileClient> stockpile,
            LabelAllocator labelAllocator)
    {
        if (!metabase.keySet().equals(stockpile.keySet())) {
            String msg = "invalid configuration of Metabase and Stockpile clients: " +
                    metabase.keySet() + " != " + stockpile.keySet();
            throw new IllegalStateException(msg);
        }

        Map<String, MetricsClient> clientByDest = new HashMap<>(metabase.size());
        for (String dest : metabase.keySet()) {
            MetricsClient client = create(dest, metabase.get(dest), stockpile.get(dest), labelAllocator);
            clientByDest.put(dest, client);
        }

        return (clientByDest.size() == 1)
                ? clientByDest.values().iterator().next()
                : new CrossDcMetricsClient(clientByDest);
    }

    public static MetricsClient create(
            TMetabaseClientConfig metabaseConfig,
            Map<String, MetabaseClient> metabase,
            Map<String, StockpileClient> stockpile,
            @Nullable DataProxyClient dataProxy,
            MetricRegistry metricRegistry,
            LabelAllocator labelAllocator,
            String clientId,
            FeatureFlagsHolder flagsHolder,
            MetricsClientMetrics metricsClientMetrics)
    {
        MetricsClient backend = create(metabase, stockpile, labelAllocator);

        @Nullable
        MetabaseFindCache metabaseFindCache = null;
        if (metabaseConfig.hasFindCacheConfig()) {
            metabaseFindCache = new MetabaseFindCacheImpl(backend, metricRegistry,
                    FindCacheOptions.create(metabaseConfig.getFindCacheConfig()));
        }

        @Nullable
        UnrollClient unrollClient = null;
        if (metabaseConfig.hasUniqueLabelsCacheConfig()) {
            Duration expireTtl = UnrollClientOptions.create(metabaseConfig.getUniqueLabelsCacheConfig()).getExpireTtl();
            unrollClient = new UnrollClientImpl(expireTtl, backend, Ticker.systemTicker());
        }

        if (metabaseFindCache != null || unrollClient != null) {
            backend = new CachingMetricsClientImpl(backend, metabaseFindCache, unrollClient);
        }

        if (dataProxy != null) {
            backend = new DataProxyAwareClient(dataProxy, backend, clientId, flagsHolder, metricsClientMetrics);
        }

        if (metricsClientMetrics != null) {
            backend = new MetricsClientMetricsProxy(metricsClientMetrics, backend);
        }

        return backend;
    }
}
