package ru.yandex.solomon.metrics.client;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ReadManyRequest extends AbstractRequest {
    private final List<MetricKey> keys;
    private final List<Operation> operations;
    private final long fromMillis;
    private final long toMillis;

    private ReadManyRequest(Builder builder) {
        super(builder);
        this.keys = List.copyOf(builder.keys);
        this.fromMillis = builder.fromMillis;
        this.toMillis = builder.toMillis;
        this.operations = List.copyOf(builder.operations);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public List<MetricKey> getKeys() {
        return keys;
    }

    public Interval getInterval() {
        return Interval.millis(fromMillis, toMillis);
    }

    public long getFromMillis() {
        return fromMillis;
    }

    public long getToMillis() {
        return toMillis;
    }

    public List<Operation> getOperations() {
        return operations;
    }

    @Override
    public Set<String> getDestinations() {
        var predefined = super.getDestinations();
        if (!predefined.isEmpty()) {
            return predefined;
        }

        if (keys.isEmpty()) {
            return Set.of();
        }

        Set<String> destinations = new HashSet<>(keys.get(0).getStockpileKeys().size());
        for (var key : keys) {
            for (var stockpileKey : key.getStockpileKeys()) {
                 destinations.add(stockpileKey.getDestination());
            }
        }
        return destinations;
    }

    public static class Builder extends AbstractRequest.Builder<Builder> {
        private final List<MetricKey> keys = new ArrayList<>();
        private long fromMillis;
        private long toMillis;

        private final List<Operation> operations = new ArrayList<>();

        public Builder() {
        }

        public Builder(ReadManyRequest request) {
            super(request);
            this.keys.addAll(request.getKeys());
            this.fromMillis = request.fromMillis;
            this.toMillis = request.toMillis;
            this.operations.addAll(request.operations);
        }

        public Builder addKey(MetricKey key) {
            this.keys.add(key);
            return this;
        }

        public Builder addKeys(List<MetricKey> keys) {
            this.keys.addAll(keys);
            return this;
        }

        public Builder setKeys(List<MetricKey> keys) {
            this.keys.clear();
            this.keys.addAll(keys);
            return this;
        }

        public Builder setFrom(Instant from) {
            this.fromMillis = from.toEpochMilli();
            return this;
        }

        public Builder setFromMillis(long fromMillis) {
            this.fromMillis = fromMillis;
            return this;
        }

        public Builder setToMillis(long toMillis) {
            this.toMillis = toMillis;
            return this;
        }

        public Builder setTo(Instant to) {
            this.toMillis = to.toEpochMilli();
            return this;
        }

        public Builder setInterval(Interval interval) {
            this.fromMillis = interval.getBeginMillis();
            this.toMillis = interval.getEndMillis();
            return this;
        }

        public Builder setOperations(List<Operation> operations) {
            this.operations.clear();
            this.operations.addAll(operations);
            return this;
        }

        public Builder addOperation(Operation operation) {
            this.operations.add(operation);
            return this;
        }

        public Builder addOperations(List<Operation> operations) {
            this.operations.addAll(operations);
            return this;
        }

        public ReadManyRequest build() {
            if (toMillis == 0) {
                toMillis = System.currentTimeMillis();
            }

            if (keys.isEmpty()) {
                throw new IllegalArgumentException("No specified key's to load");
            }

            return new ReadManyRequest(this);
        }
    }
}
