package ru.yandex.solomon.metrics.client;

import java.time.Instant;
import java.util.HashSet;
import java.util.Set;

import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.util.time.Interval;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadRequest extends AbstractRequest {
    private final MetricKey key;
    private final long fromMillis;
    private final long toMillis;
    private final long gridMillis;
    private final Aggregation aggregation;

    private ReadRequest(Builder builder) {
        super(builder);
        this.key = requireNonNull(builder.key);
        this.fromMillis = builder.fromMillis;
        this.toMillis = builder.toMillis;
        this.gridMillis = builder.gridMillis;
        this.aggregation = builder.aggregation;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public MetricKey getKey() {
        return key;
    }

    public long getFromMillis() {
        return fromMillis;
    }

    public long getToMillis() {
        return toMillis;
    }

    public long getGridMillis() {
        return gridMillis;
    }

    public Aggregation getAggregation() {
        return aggregation;
    }

    @Override
    public Set<String> getDestinations() {
        var predefined = super.getDestinations();
        if (!predefined.isEmpty()) {
            return predefined;
        }

        var stockpileKeys = key.getStockpileKeys();
        switch (stockpileKeys.size()) {
            case 0:
                return Set.of();
            case 1:
                return Set.of(stockpileKeys.get(0).getDestination());
            default:
                Set<String> result = new HashSet<>(stockpileKeys.size());
                for (var key : stockpileKeys) {
                    result.add(key.getDestination());
                }
                return result;
        }
    }

    public static class Builder extends AbstractRequest.Builder<Builder> {
        private MetricKey key;
        private long fromMillis;
        private long toMillis;
        private long gridMillis;
        private Aggregation aggregation = Aggregation.DEFAULT_AGGREGATION;

        public Builder setKey(MetricKey key) {
            this.key = key;
            return this;
        }

        public Builder setFrom(Instant from) {
            this.fromMillis = from.toEpochMilli();
            return this;
        }

        public Builder setFromMillis(long fromMillis) {
            this.fromMillis = fromMillis;
            return this;
        }

        public Builder setToMillis(long toMillis) {
            this.toMillis = toMillis;
            return this;
        }

        public Builder setTo(Instant to) {
            this.toMillis = to.toEpochMilli();
            return this;
        }

        public Builder setInterval(Interval interval) {
            this.fromMillis = interval.getBeginMillis();
            this.toMillis = interval.getEndMillis();
            return this;
        }

        public Builder setGridMillis(long gridMillis) {
            this.gridMillis = gridMillis;
            return this;
        }

        public Builder setAggregation(Aggregation aggregation) {
            this.aggregation = aggregation;
            return this;
        }

        public ReadRequest build() {
            return new ReadRequest(this);
        }
    }
}
