package ru.yandex.solomon.metrics.client;

import java.nio.channels.ClosedChannelException;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import io.grpc.Status;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.stockpile.api.EStockpileStatusCode;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class StockpileStatus {
    public static StockpileStatus OK = StockpileStatus.fromCode(EStockpileStatusCode.OK, "");

    private final EStockpileStatusCode code;
    @Nullable
    private final String description;

    public StockpileStatus(EStockpileStatusCode code, @Nullable String description) {
        this.code = code;
        this.description = description;
    }

    public static StockpileStatus fromCode(EStockpileStatusCode code, @Nullable String description) {
        return new StockpileStatus(code, description);
    }

    public static StockpileStatus fromThrowable(Throwable t) {
        var status = Status.fromThrowable(t);
        switch (status.getCode()) {
            case RESOURCE_EXHAUSTED:
                return new StockpileStatus(EStockpileStatusCode.RESOURCE_EXHAUSTED, status.toString());
            case DEADLINE_EXCEEDED:
                return new StockpileStatus(EStockpileStatusCode.DEADLINE_EXCEEDED, status.toString());
            case UNAVAILABLE:
            case CANCELLED:
                return new StockpileStatus(EStockpileStatusCode.NODE_UNAVAILABLE, status.toString());
            case FAILED_PRECONDITION:
            case INVALID_ARGUMENT:
                return new StockpileStatus(EStockpileStatusCode.INVALID_REQUEST, status.toString());
            case UNKNOWN:
                if (status.getCause() instanceof ClosedChannelException) {
                    return new StockpileStatus(EStockpileStatusCode.NODE_UNAVAILABLE, status.toString());
                }
                // fall through
            default:
                return new StockpileStatus(EStockpileStatusCode.INTERNAL_ERROR, Throwables.getStackTraceAsString(t));
        }
    }

    public EStockpileStatusCode getCode() {
        return code;
    }

    public String getDescription() {
        return description;
    }

    @Override
    public String toString() {
        if (StringUtils.isEmpty(description)) {
            return code.toString();
        } else {
            return code + ": " + description;
        }
    }
}
