package ru.yandex.solomon.metrics.client;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.ByteString;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.archive.serializer.MetricHeaderSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.ConcatAggrGraphDataIterable;
import ru.yandex.stockpile.api.TCompressedReadResponse;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.model.point.column.StockpileColumns.ensureColumnSetValidIfNotEmpty;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class TimeSeriesCodec {

    @Nonnull
    public static MetricArchiveImmutable decode(StockpileFormat format, TimeSeries.Chunk chunk) {
        if (chunk.getPointCount() == 0) {
            return MetricArchiveImmutable.empty;
        }

        return decode(format, chunk.getContent());
    }

    @Nonnull
    public static MetricArchiveImmutable decode(StockpileFormat format, ByteString chunk) {
        StockpileDeserializer deserializer = new StockpileDeserializer(chunk);
        var archive = MetricArchiveNakedSerializer.serializerForFormatSealed(format).deserializeToEof(deserializer);
        ensureColumnSetValidIfNotEmpty(archive.getType(), archive.columnSetMask());
        return archive;
    }

    public static MetricType decodeType(TimeSeries timeSeries) {
        if (timeSeries.getChunksCount() == 0) {
            return MetricType.METRIC_TYPE_UNSPECIFIED;
        }

        StockpileDeserializer deserializer = new StockpileDeserializer(timeSeries.getChunks(0).getContent());
        MetricHeader header = MetricHeaderSerializer.I.deserializeWithLength(deserializer);
        return header.getType();
    }

    public static AggrGraphDataIterable sequenceDecode(TCompressedReadResponse response) {
        StockpileFormat format = StockpileFormat.byNumber(response.getBinaryVersion());
        return sequenceDecode(format, response.getChunksList());
    }

    public static AggrGraphDataIterable sequenceDecode(TimeSeries compressed) {
        StockpileFormat format = StockpileFormat.byNumber(compressed.getFormatVersion());
        return sequenceDecode(format, compressed);
    }

    private static AggrGraphDataIterable sequenceDecode(StockpileFormat format, TimeSeries compressed) {
        if (compressed.getChunksCount() == 0) {
            return MetricArchiveImmutable.empty;
        }

        if (compressed.getChunksCount() == 1) {
            return decode(format, compressed.getChunks(0));
        }

        return compressed.getChunksList()
            .stream()
            .map(chunk -> decode(format, chunk))
            .collect(collectingAndThen(toList(), ConcatAggrGraphDataIterable::of));
    }

    private static AggrGraphDataIterable sequenceDecode(StockpileFormat format, List<TimeSeries.Chunk> chunks) {
        if (chunks.isEmpty()) {
            return MetricArchiveImmutable.empty;
        }

        if (chunks.size() == 1) {
            return decode(format, chunks.get(0));
        }

        return chunks.stream()
                .map(chunk -> decode(format, chunk))
                .collect(collectingAndThen(toList(), ConcatAggrGraphDataIterable::of));
    }
}
