package ru.yandex.solomon.metrics.client.cache;

import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillCloseWhenClosed;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.metrics.client.FindResponse;
import ru.yandex.solomon.metrics.client.LabelNamesRequest;
import ru.yandex.solomon.metrics.client.LabelNamesResponse;
import ru.yandex.solomon.metrics.client.LabelValuesRequest;
import ru.yandex.solomon.metrics.client.LabelValuesResponse;
import ru.yandex.solomon.metrics.client.MetricNamesRequest;
import ru.yandex.solomon.metrics.client.MetricNamesResponse;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.ReadManyResponse;
import ru.yandex.solomon.metrics.client.ReadRequest;
import ru.yandex.solomon.metrics.client.ReadResponse;
import ru.yandex.solomon.metrics.client.ResolveManyRequest;
import ru.yandex.solomon.metrics.client.ResolveManyResponse;
import ru.yandex.solomon.metrics.client.ResolveManyWithNameRequest;
import ru.yandex.solomon.metrics.client.ResolveOneRequest;
import ru.yandex.solomon.metrics.client.ResolveOneResponse;
import ru.yandex.solomon.metrics.client.ResolveOneWithNameRequest;
import ru.yandex.solomon.metrics.client.UniqueLabelsRequest;
import ru.yandex.solomon.metrics.client.UniqueLabelsResponse;
import ru.yandex.solomon.metrics.client.combined.CombinedCall;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyResponse;
import ru.yandex.solomon.selfmon.AvailabilityStatus;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class CachingMetricsClientImpl implements MetricsClient {
    private final MetricsClient backend;

    @Nullable
    private final MetabaseFindCache findCache;
    @Nullable
    private final UnrollClient unrollClient;

    public CachingMetricsClientImpl(
            @WillCloseWhenClosed MetricsClient backend,
            @Nullable MetabaseFindCache findCache,
            @Nullable UnrollClient unrollClient)
    {
        this.backend = backend;
        this.findCache = findCache;
        this.unrollClient = unrollClient;
    }

    public CachingMetricsClientImpl(@WillCloseWhenClosed MetricsClient backend, MetabaseFindCache findCache) {
        this(backend, findCache, new UnrollClientStub(backend));
    }

    public CachingMetricsClientImpl(@WillCloseWhenClosed MetricsClient backend) {
        this(backend, new MetabaseFindCacheStub(backend));
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        if (findCache != null) {
            return findCache.find(request.getSelectors(), request.getLimit(), request.getSoftDeadline(), request.getDeadline());
        }
        return backend.find(request);
    }

    @Override
    public CompletableFuture<UniqueLabelsResponse> uniqueLabels(UniqueLabelsRequest request) {
        if (unrollClient != null) {
            return unrollClient.uniqueLabels(request);
        }
        return backend.uniqueLabels(request);
    }

    // Default findAndReadMany calls this::find, not backend::find as backend::findAndReadMany does.
    // So leave the default implementation

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        return backend.resolveOne(request);
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOneWithName(ResolveOneWithNameRequest request) {
        return backend.resolveOneWithName(request);
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        return backend.resolveMany(request);
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveManyWithName(ResolveManyWithNameRequest request) {
        return backend.resolveManyWithName(request);
    }

    @Override
    public CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request) {
        return backend.metricNames(request);
    }

    @Override
    public CompletableFuture<LabelNamesResponse> labelNames(LabelNamesRequest request) {
        return backend.labelNames(request);
    }

    @Override
    public CompletableFuture<LabelValuesResponse> labelValues(LabelValuesRequest request) {
        return backend.labelValues(request);
    }

    @Override
    public CompletableFuture<ReadResponse> read(ReadRequest request) {
        return backend.read(request);
    }

    @Override
    public CompletableFuture<ReadManyResponse> readMany(ReadManyRequest request) {
        return backend.readMany(request);
    }

    @Override
    public CompletableFuture<FindAndReadManyResponse> findAndReadMany(FindAndReadManyRequest request) {
        return CombinedCall.defaultFindAndReadMany(this, request);
    }

    @Override
    public AvailabilityStatus getMetabaseAvailability() {
        return backend.getMetabaseAvailability();
    }

    @Override
    public AvailabilityStatus getStockpileAvailability() {
        return backend.getStockpileAvailability();
    }

    @Override
    public Stream<Labels> metabaseShards(String destination, Selectors selector) {
        return backend.metabaseShards(destination, selector);
    }

    @Override
    public String getStockpileHostForShardId(String destination, int shardId) {
        return backend.getStockpileHostForShardId(destination, shardId);
    }

    @Override
    public Collection<String> getDestinations() {
        return backend.getDestinations();
    }

    @Override
    public void close() {
        backend.close();
    }
}
