package ru.yandex.solomon.metrics.client.cache;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.config.OptionalSet;
import ru.yandex.solomon.config.protobuf.metabase.client.TMetabaseClientConfig;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class FindCacheOptions {
    private final Duration refreshInterval;
    private final Duration expireTtl;
    private final long maxSize;

    public FindCacheOptions(Duration refreshInterval, Duration expireTtl, long maxSize) {
        this.refreshInterval = refreshInterval;
        this.expireTtl = expireTtl;
        this.maxSize = maxSize;
    }

    public Duration getRefreshInterval() {
        return refreshInterval;
    }

    public Duration getExpireTtl() {
        return expireTtl;
    }

    public long getMaxSize() {
        return maxSize;
    }

    public static FindCacheOptions create(TMetabaseClientConfig.FindCacheOptions proto) {
        Builder builder = FindCacheOptions.newBuilder();

        OptionalSet.setLong(builder::setMaxSize, proto.getMaxSize());
        OptionalSet.setTime(builder::setRefreshInterval, proto.getRefreshInterval());
        OptionalSet.setTime(builder::setExpireTtl, proto.getExpireTtl());

        return builder.build();
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private Duration refreshInterval = Duration.ofMinutes(10);
        @Nullable
        private Duration expireTtl = null; // default to refreshInterval
        long maxSize = 0;

        public Builder setRefreshInterval(long value, TimeUnit unit) {
            this.refreshInterval = Duration.ofMillis(unit.toMillis(value));
            return this;
        }

        public Builder setExpireTtl(long value, TimeUnit unit) {
            this.expireTtl = Duration.ofMillis(unit.toMillis(value));
            return this;
        }

        public Builder setMaxSize(long maxSize) {
            this.maxSize = maxSize;
            return this;
        }

        public FindCacheOptions build() {
            return new FindCacheOptions(
                    refreshInterval,
                    (expireTtl == null) ? refreshInterval : expireTtl,
                    maxSize
            );
        }
    }
}
