package ru.yandex.solomon.metrics.client.cache;

import java.time.Duration;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import com.google.common.base.Ticker;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.UniqueLabelsRequest;
import ru.yandex.solomon.metrics.client.UniqueLabelsResponse;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.metrics.client.CrossDcResponseMerger.mergeUniqueLabelsResponses;

/**
 * @author Vladimir Gordiychuk
 */
public class UnrollClientImpl implements UnrollClient {
    private final MetricsClient client;
    private final Cache<Key, UniqueLabelsResponse> cache;

    public UnrollClientImpl(Duration expireTime, MetricsClient client, Ticker ticker) {
        this.client = client;
        this.cache = CacheBuilder.newBuilder()
            .expireAfterWrite(expireTime.toMillis(), TimeUnit.MILLISECONDS)
            .ticker(ticker)
            .build();
    }

    @Override
    public CompletableFuture<UniqueLabelsResponse> uniqueLabels(UniqueLabelsRequest request) {
        return client.getDestinations()
            .stream()
            .map(dest -> {
                var key = new Key(dest, request.getSelectors(), request.getLabels());
                var cached = cache.getIfPresent(key);
                if (cached != null) {
                    return CompletableFuture.completedFuture(cached);
                }

                return dcFind(key, request);
            })
            .collect(collectingAndThen(toList(), CompletableFutures::allOf))
            .thenApply(list -> mergeUniqueLabelsResponses("", list));
    }

    private CompletableFuture<UniqueLabelsResponse> dcFind(Key key, UniqueLabelsRequest request) {
        return client.uniqueLabels(request.toBuilder()
            .setDestination(key.dc)
            .setProducer(RequestProducer.SYSTEM)
            .build())
            .thenApply(response -> {
                if (response.isOk()) {
                    cache.put(key, response);
                }

                return response;
            });
    }

    private static class Key {
        private final String dc;
        private final Selectors selectors;
        private final Set<String> groupKeys;

        public Key(String dc, Selectors selectors, Set<String> groupKeys) {
            this.dc = dc;
            this.selectors = selectors;
            this.groupKeys = groupKeys;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            Key key = (Key) o;

            if (!dc.equals(key.dc)) return false;
            if (!selectors.equals(key.selectors)) return false;
            return groupKeys.equals(key.groupKeys);
        }

        @Override
        public int hashCode() {
            int result = dc.hashCode();
            result = 31 * result + selectors.hashCode();
            result = 31 * result + groupKeys.hashCode();
            return result;
        }
    }
}
