package ru.yandex.solomon.metrics.client.combined;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationAggregationSummary;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.math.protobuf.OperationDropTimeSeries;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.metrics.client.FindResponse;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.ReadManyResponse;
import ru.yandex.solomon.metrics.client.exceptions.TooManyMetricsLoadedBySelectors;
import ru.yandex.solomon.metrics.client.exceptions.TooManyMetricsReturnedPerGraph;
import ru.yandex.solomon.model.MetricKey;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class CombinedCall {
    private static final Logger logger = LoggerFactory.getLogger(CombinedCall.class);

    public static CompletableFuture<FindAndReadManyResponse> defaultFindAndReadMany(
            MetricsClient metricsClient,
            FindAndReadManyRequest request)
    {
        var findRequest = FindRequest.newBuilder()
                .setDeadline(request.getDeadline())
                .setSoftDeadline(request.getSoftDeadline())
                .setLimit(request.getMetabaseLimit())
                .setSelectors(request.getSelectors())
                .setUseNewFormat(request.isUseNewFormat())
                .setProducer(request.getProducer())
                .setDestination(request.getDestination());

        return metricsClient.find(findRequest.build())
                .thenCompose(findResponse -> handleFindResponse(metricsClient, request, findResponse));
    }

    private static CompletableFuture<FindAndReadManyResponse> handleFindResponse(
            MetricsClient metricsClient,
            FindAndReadManyRequest request,
            FindResponse findResponse)
    {
        if (!findResponse.isOk()) {
            return CompletableFuture.completedFuture(new FindAndReadManyResponse(findResponse.getStatus()));
        }

        List<MetricKey> metricKeys = findResponse.getMetrics();

        OldModeResult oldModeResult = ensureMetabaseLimitsAndDeduceMode(
                findResponse,
                request.isOldMode(),
                request.getMetabaseLimit(),
                request.getRequestType(),
                request.getSelectors(),
                request.getOperations());

        if (metricKeys.isEmpty()) {
            return CompletableFuture.completedFuture(new FindAndReadManyResponse(List.of(), oldModeResult));
        }

        ReadManyRequest.Builder builder = ReadManyRequest.newBuilder()
                .addKeys(metricKeys)
                .setDeadline(request.getDeadline())
                .setSoftDeadline(request.getSoftReadDeadline())
                .setFromMillis(request.getFromMillis())
                .setToMillis(request.getToMillis())
                .setProducer(request.getProducer())
                .setDestination(request.getDestination())
                .addOperations(request.getOperations())
                .setSubjectId(request.getSubjectId());

        if (oldModeResult.isSummary() && !hasSummaryOperation(request.getOperations())) {
            addOnlySummaryOperation(builder::addOperation);
        }

        ReadManyRequest readManyRequest = builder.build();

        return readMetricsByKeys(
                metricsClient,
                metricKeys,
                request.getOperations(),
                readManyRequest)
            .thenApply(readResponse -> ensureResponseIsOk(readResponse, oldModeResult))
            .whenComplete((response, throwable) -> {
                if (throwable != null) {
                    return;
                }
                if (response.isOk()) {
                    if (response.getMetrics().isEmpty()) {
                        logger.warn("Empty findAndReadMany result for {} {}", request, findResponse);
                    }
                }
            });
    }

    private static final List<Aggregation> ALL_SUMMARIES = List.of(
            Aggregation.MIN,
            Aggregation.MAX,
            Aggregation.AVG,
            Aggregation.SUM,
            Aggregation.LAST
    );

    // TODO(uranix): move this method to DataClient once oldMode is gone
    public static void addOnlySummaryOperation(Consumer<Operation> consumer) {
        consumer.accept(Operation.newBuilder()
                .setSummary(OperationAggregationSummary.newBuilder()
                        .addAllAggregations(ALL_SUMMARIES))
                .build());

        consumer.accept(Operation.newBuilder()
                .setDropTimeseries(OperationDropTimeSeries.getDefaultInstance())
                .build());
    }

    private static boolean hasSummaryOperation(List<Operation> operations) {
        return operations.stream().anyMatch(Operation::hasSummary);
    }

    public static CompletableFuture<ReadManyResponse> readMetricsByKeys(
            MetricsClient metricsClient,
            /*@Nonempty*/ List<MetricKey> metricKeys,
            List<Operation> operations,
            ReadManyRequest request)
    {
        int expectedMetrics = operations.stream()
                .filter(Operation::hasTop)
                .map(Operation::getTop)
                .mapToInt(op -> Math.min(op.getLimit(), metricKeys.size()))
                .findFirst()
                .orElse(metricKeys.size());

        long gridMillis = operations.stream()
                .filter(Operation::hasDownsampling)
                .map(Operation::getDownsampling)
                .mapToLong(OperationDownsampling::getGridMillis)
                .findFirst()
                .orElse(0);

        boolean timeseriesPresent = operations.stream()
                .noneMatch(Operation::hasDropTimeseries);

        if (timeseriesPresent) {
            DataLimits.ensureResponseSizeLimit(request.getInterval(), gridMillis, expectedMetrics);
        }

        return TopOptimization.readMany(metricsClient, request);
    }

    /*
     This cratch is about to support old mode for autographs:
     - limit every request for more than 10000 metrics and show "truncated" flag in data response
     - show timeseries as summary for more than 640 metrics and show "summary" flag in data response
     See SOLOMON-5053 for more details.
     */
    public static OldModeResult ensureMetabaseLimitsAndDeduceMode(
            FindResponse metabaseResponse,
            boolean isOldMode,
            int metabaseReqLimit,
            DataRequestType dataRequestType,
            Selectors selectors,
            List<Operation> operations)
    {
        List<MetricKey> metrics = metabaseResponse.getMetrics();
        int metricCount = metrics.size();
        boolean truncated = metabaseResponse.isTruncated();
        boolean tooManyMetrics = metricCount > DataLimits.MAX_METRICS_COUNT;
        boolean seriesDropped = operations.stream().anyMatch(Operation::hasDropTimeseries);

        if (isOldMode) {
            if (dataRequestType == DataRequestType.COMPLEX && truncated) {
                throw new TooManyMetricsLoadedBySelectors(selectors, metabaseReqLimit);
            }

            boolean summary = dataRequestType == DataRequestType.SIMPLE && tooManyMetrics;

            return new OldModeResult(truncated, summary || seriesDropped);
        } else {
            if (truncated) {
                throw new TooManyMetricsLoadedBySelectors(selectors, metabaseReqLimit);
            }

            if (dataRequestType == DataRequestType.SIMPLE && tooManyMetrics) {
                throw new TooManyMetricsReturnedPerGraph(metricCount, DataLimits.MAX_METRICS_COUNT);
            }

            return new OldModeResult(false, seriesDropped);
        }
    }

    private static FindAndReadManyResponse ensureResponseIsOk(ReadManyResponse readManyResponse, OldModeResult oldModeResult) {
        if (!readManyResponse.isOk()) {
            return new FindAndReadManyResponse(readManyResponse.getStatus());
        }

        return new FindAndReadManyResponse(readManyResponse.getMetrics(), oldModeResult);
    }
}
